class EntryTypeahead {
    constructor(options) {
		this.options = $.extend(true, 
			{
				suggestionsLimit: 10,
				cache: false,
				highlight: true,
				minQueryLength: 2
			}, 
			options);
			
        this.typeaheadSources = {};
    }
    
    initTypeahead(parent) {
        let _this = this;

        $(parent).find(".typeahead").each(function() {
            const vocabulary = $(this).data("vocabulary");
            const entity = $(this).data("vocabulary-entity");
            const strict = $(this).data("strict");

            const sourceId = (entity ? "e_" : "v_") + vocabulary;
            const sourceUrl = 'api/v1/' + (entity ? "e/" : "v/") + vocabulary + '/search';
			const getUrl = 'api/v1/' + (entity ? "e/" : "v/") + vocabulary + '/e';

			$(this).data("geturl", getUrl);
			
			const displayField = entity ? "entityId" : "key";
			$(this).data("geturl", getUrl);
			$(this).data("displayField", displayField);

			_this.createTypeahead(this, sourceId, sourceUrl, displayField, strict);
			_this.setTypeaheadValue(this);
        });
    }

    createTypeahead(element, sourceId, sourceUrl, displayField, strict) {
		const _this = this;
		
		let typeaheadTemplates = {
			 empty: function() { return _this.renderTypeaheadEmpty(strict) },
             suggestion: function(data) { return _this.renderTypeaheadSuggestionInline(data) }
		}
		
		if (!strict) {
			typeaheadTemplates.footer = function() { return _this.renderTypeaheadEmpty() };
		}
				
		$(element).typeahead(
            {
                highlight: _this.options.highlight,
                minLength: _this.options.minQueryLength,
            }, {
                name: sourceId,
                source: _this.getOrCreateTypeaheadSource(sourceId, sourceUrl),
                display: displayField,
                limit: _this.options.suggestionsLimit*2, // Fix(2) for https://github.com/twitter/typeahead.js/issues/1686#issuecomment-546655910
                templates: typeaheadTemplates
            }
		);
		
		this.bindEvents(element);
	}

	bindEvents(element) {
		const _this = this;

		// Reset icon when changing value
		$(element).on('typeahead:asynccancel', function() {
			$(this).closest(".input-group").find(".input-group-text").html('<i class="bi bi-search"></i>');
		});

		// Lost focus and content has changed
		$(element).on('typeahead:change', function(ev, suggestion) {
			_this.setTypeaheadValue(element);
		});
				
		// Suggestion selected
		$(element).on('typeahead:select', function(ev, suggestion) { _this.handleSuggestionSelected(this, suggestion)});
		$(element).on('typeahead:autocomplete', function(ev, suggestion) { _this.handleSuggestionSelected(this, suggestion)});
		
		// Loading
		$(element).on('typeahead:asyncrequest', function() {
			$(this).closest(".input-group").find(".input-group-text").html('<div class="spinner-border spinner-border-sm" role="status"><span class="visually-hidden">Loading...</span></div>');
		});
		$(element).on('typeahead:asyncreceive', function() {
			$(this).closest(".input-group").find(".input-group-text").html('<i class="bi bi-search"></i>');
		});
		$(element).on('typeahead:asyncreceive', function() {
			$(this).closest(".input-group").find(".input-group-text").html('<i class="bi bi-search"></i>');
		});
	};
	
	setTypeaheadValue(element) {
		const _this = this;
		const field = $(element).data("displayField");
		const val = $(element).typeahead('val');

		if (val === $(element).closest(".property-value-list-item, .property-container").find(".selected-suggestion .alert").data("selected-val")) {
			return;
		}
		
		$(element).closest(".property-value-list-item, .property-container").find(".selected-suggestion").text('');
		
		if (val.length==0) {
			return;
		}
		
		let requestData = { "ui" : true };
		requestData[field] = val;
		
		$.ajax({
			type: "POST",
			contentType: "application/json; charset=UTF-8",
			dataType: "json",
			url: __util.composeUrl($(element).data("geturl")),
			data: JSON.stringify(requestData),
			success: function(data) {
				 _this.handleSuggestionSelected(element, data.item);
			}
		});
	};
	
	handleSuggestionSelected(element, suggestion, freevalue) {
		if ($(element).hasClass("language-typeahead")) {
			return;
		}
		const val = $(element).typeahead('val');
		const prop = $(element).closest(".property-container").data("property-container");

		const listItem = $("<div class='mb-1 property-value-list-item'>");
		
		const flexContainer = $("<div class='d-flex'>");
		
		const hiddenInput = $("<input type='hidden' class='form-body-element' name='" + prop + "' value='" + val + "'>"); 
		flexContainer.append(hiddenInput);
		
		const label = $("<label class='form-control-plaintext ms-2 flex-grow-1'>");
		if (freevalue) {
			label.append("<span class='badge rounded-pill text-bg-primary'>freetext</span> ");
			label.append(suggestion);
		} else if (suggestion.entityId) {
			label.append("<span class='badge rounded-pill text-bg-secondary'>entity</span> ");
			label.append(this.renderTypeaheadSuggestionInline(suggestion));
		} else {
			label.append("<span class='badge rounded-pill text-bg-secondary'>entry</span> ");
			label.append(this.renderTypeaheadSuggestionInline(suggestion));
		}

		flexContainer.append(label);

		flexContainer.append('<div class="text-nowrap align-self-center">' +
			'<button onclick="entityForm.findList(this).pushEntryUp(this); return false;" type="button" class="btn btn-link btn-valuelist-push-up"><i class="bi bi-arrow-up"></i></button> ' +
			'<button onclick="entityForm.findList(this).pushEntryDown(this); return false;" type="button" class="btn btn-link btn-valuelist-push-down"><i class="bi bi-arrow-down"></i></button> ' +
			'<button onclick="entityForm.findList(this).removeEntry(this); return false;" type="button" class="btn btn-link btn-valuelist-push-remove text-danger"><i class="bi bi-x-lg"></i></button>' +
		'</div>');

		listItem.append(flexContainer);

	  	$(element).closest(".property-container").children(".invalid-feedback").before(listItem);
	  	
	  	$(element).typeahead('destroy');
		$(element).closest(".property-value-list-item").remove();
		
		if (entityForm) {
			entityForm.findList(hiddenInput).sort();
		}
	}

    getOrCreateTypeaheadSource(sourceId, sourceUrl) {
		const _this = this;
		if (!this.typeaheadSources.hasOwnProperty(sourceId)) {
			this.typeaheadSources[sourceId] = new Bloodhound({
				suggestionsLimit: _this.options.suggestionsLimit,
				datumTokenizer: Bloodhound.tokenizers.whitespace,
				queryTokenizer: Bloodhound.tokenizers.whitespace,
				remote: {
					url: __util.composeUrl(sourceUrl),
					cache: _this.options.cache,
					prepare: function(query, settings) {
						settings.type = "POST";
						settings.contentType = "application/json; charset=UTF-8";
						settings.data = JSON.stringify({ "query" : query, "ui" : true });
						return settings;
					},
					transform: function(response) {
						// Fix(1) for https://github.com/twitter/typeahead.js/issues/1686#issuecomment-546655910
						if (response.items.length > _this.options.suggestionsLimit) {
							return response.items.slice(0, _this.options.suggestionsLimit);
						} else {
							return response.items;
						}
					}
				}
			});
		}
		return this.typeaheadSources[sourceId];
	}
    
     
    renderTypeaheadSuggestionInline(data) {
		let displayVal = "";
		if (data._display) {
			 displayVal = data._display;
		} else {
			for (let i=0; i<data._view.length; i++) {
				let prop = data._view[i];
	            if (prop.value === undefined || prop.value === null) {
	                continue;
	            }
	            displayVal = displayVal + this.renderTypeaheadSuggestionValue(prop);
	            if (i<data._view.length-1) {
					displayVal += ", ";
				}
	        }
        }
        return "<span>" + displayVal + "</span>";
	}
    
    
    renderTypeaheadSuggestionValue(prop) {
        if (prop.value.values) {
            let value = "";
            for (let j = 0; j < prop.value.values.length; j++) {
                value = value + prop.value.values[j].value;
                if (j < prop.value.values.length - 1) {
                    value = value + ", ";
                }
            }
            return value;
        } else {
            return prop.value.value;
        }
    }
    
    renderTypeaheadEmpty(strict) {
		const _this = this;
		let empty = $("<div class='empty-message'>");
		if (!strict) {
			$(empty)
				.addClass("tt-suggestion tt-selectable")
				.text(__translator.translate("view.entity_form.typeahead.no_results_add_free"))
				.on("click", function(event) {
					_this.selectTypeaheadEmpty(this);
					event.preventDefault();
				});
		} else {
			$(empty).text(__translator.translate("view.entity_form.typeahead.no_results"));
		}
		return empty;
	}
	
	selectTypeaheadEmpty(empty) {
		const typeahead = $(empty).closest(".twitter-typeahead").find(".typeahead.tt-input");
		const val = $(typeahead).typeahead('val');
		this.handleSuggestionSelected(typeahead, val, true)
	}
}