/**
 * 	BaseTable: superclass for all editors handling dataTables
 * ==========================================================================================
 * 	Notes:
 * 		- All filter options that are indended to trigger a reload of the table
 * 		  need to hold a class 'editor-option'
 * 		- Defaults for bProcessing, sAjaxSource, bAutoWidth and the fnDrawCallback
 * 		  are set here but can be overridden when initializing the actual table
 * 		- Tooltips in the data are identified by the class 'hint-tooltip' and
 * 		  loaded after every refresh of the editor
 */

// Default classes for buttons
$.fn.dataTable.Buttons.defaults.dom.button.className = 'btn';

class BaseTable {
    constructor(options, translations) {
        // Explicit accessor for base properties in the editors 
        this.table = null;
        this.error = false;

		this.options = $.extend(true,
            {
                containerSelector: null,
                refreshInterval: __properties.refreshIntervalMs,
                cyclicRefresh: __properties.refreshViews,
                buttons: [],
                includeExportButtons: false,
                includeSearchPane: false,
                includeColumnReorder: false,
                clickRowEvent: false,
                useBootstrapIcons: true,
                showInfo: true,
                showFind: true,
                paginate: true,
                rowGroup: undefined,
                actionIcons: {
                    export: 'bi-download',
                    import: 'bi-upload',
                    copy: 'bi-clipboard',
                    csv: 'bi-filetype-csv',
                    print: 'bi-printer',
                    add: 'bi-plus-lg',
                    filter: 'bi-funnel',
                    columns: 'bi-columns',
                    remove: 'bi-x-lg',
                    column_visibility: 'bi-eye',
                    restore_columns: 'bi-columns-gap',
                    show_all_columns: 'bi-layout-three-columns',
                    fixed_columns: 'bi-layout-sidebar-inset',
                }
            },
            options);

        this.translations = [
            "view.error.generic_ajax.head",
            "view.error.generic_ajax.body_code",

			"view.data_tables.btn.add",
            "view.data_tables.btn.print",
            "view.data_tables.btn.copy",
            "view.data_tables.btn.copy.messageTitle",
            "view.data_tables.btn.copy.copySuccess.1",
            "view.data_tables.btn.copy.copySuccess.n",
            "view.data_tables.btn.csv",
            "view.data_tables.btn.export",
            "view.data_tables.btn.import",
            "view.data_tables.btn.filter",
            "view.data_tables.btn.columns",
            "view.data_tables.btn.column_visibility",
            "view.data_tables.btn.column_visibility.control_message",
            "view.data_tables.btn.restore_columns",
            "view.data_tables.btn.show_all_columns",

            "view.data_tables.btn.filter",
            "view.data_tables.search_panes.title._",
            "view.data_tables.search_panes.title.0",
            "view.data_tables.search_panes.title.1",
            "view.data_tables.search_panes.count",
            "view.data_tables.search_panes.count_filtered",
            "view.data_tables.search_panes.collapse_all",
            "view.data_tables.search_panes.expand_all",
            "view.data_tables.search_panes.load_message",
            "view.data_tables.search_panes.empty_panes",
            "view.data_tables.search_panes.empty_message",
            "view.data_tables.search_panes.clear_all",
            "view.data_tables.btn.fixed_columns",
            
            "view.model.freetext",
            "view.model.entity"
        ];
        if (translations) {
			this.translations.push(translations);
		}
    }
    
    initTranslations(callback) {
		// This is called sync
		if (this.translations && this.translations instanceof Array && this.translations.length > 0) {
            __translator.addTranslations(this.translations);
            __translator.getTranslations();
        }
        if (callback != undefined && typeof callback == 'function') {
            callback();
        }
    }
    
    applyBaseSettings() {
		this.addExportButtons(this.options.buttons);
        this.addColumnsButtons(this.options.buttons);
        this.addSearchPaneButton(this.options.buttons);
        
        this.setBaseSettings(this.options.buttons);
	}
    
    setBaseSettings(tableButtons) {
        const _this = this;
        
        let dom = "<'d-flex'";
        if (this.options.showInfo) {
			dom += "<'flex-grow-1' i>";
		}
		if (this.options.showFind) {
			dom += "<'me-2'f>";
		}
		if (tableButtons.length>0) {
			dom += "<' text-end'B>";
		}
		dom += ">";
        if (this.options.includeSearchPane) {
			dom += "<'search-panes-container d-none'P>";
		}
        dom += "<'row'<'col-12 data-tables-table'tr>>";
        
        if (this.options.paginate) {
			 dom += "<'row'<'col-5'l><'col-7'p>>";
		}
        
        // Setting defaults for the datatables as used in the project
        this.baseSettings = {
            dom: dom,
            buttons: tableButtons,
            language: this.getLanguage(),
            autoWidth: true,
            colReorder: this.options.includeColumnReorder,
            responsive: false,
            processing: false,
            statesave: true,
            searchPanes: {
                initCollapsed: true
            },
            // Selectable rows are a TODO and simplify deletion and mass edit
            /*select: {
                style: 'os',
                //selector: 'td:first-child'
            },*/
            drawCallback: function(oSettings) {
                if (_this.handleGrouping !== undefined) {
                    _this.handleGrouping(this, oSettings);
                }
                _this.handleRefresh(oSettings);
            },
            initComplete: function(settings, json) {
                $(_this.options.containerSelector + " thead").show();
                if (_this.options.url) {
                	_this.cycleRefresh();
                }
            }
        };
        if (this.options.url) {
			this.baseSettings.ajax = {
                url: this.options.url,
                error: function(xhr, textStatus, error) { _this.handleAjaxError(xhr, textStatus, error); },
                dataSrc: "items"
            }
		}
    }
    
    createTable() {
		let columns;
		if (this.options.url) {
        	columns = this.createDynamicColumns();
        }
        this.createDatatable(columns);
    }
    
    createDatatable(columnDefs) {
		const _this = this;
        this.table = $(this.options.selector).DataTable($.extend(true, {
            order: [[1, "asc"]],
            columnDefs: columnDefs,
            info: this.showInfo,
		    paging: this.paginate,
            rowGroup: this.options.rowGroup
        }, this.baseSettings));
        this.assignTableEvents();
	}
	
    createDynamicColumns(prefixColumns) {
        const _this = this;
        const columnDefs = [];
        if (prefixColumns) {
            columnDefs.push(prefixColumns);
        }

        let columnIndex = columnDefs.length;
        $(this.options.selector).find("thead th").each(function() {
            let label = $(this).data("property-label");
            let special = $(this).data("special-column");            
            if (!label && !special) {
                return;
            }
            let def = {
                targets: [columnIndex++],
                visible: columnIndex <= _this.options.initMaxColumns || special || _this.options.initMaxColumns<=0,
                data: function(row, type, val, meta) {
                    if (label) {
						let value = _this.renderValueColumn(label, row, type, val, meta);
						// Special treatment in inheriting table
						if (value) {
							return value;
						}
	                    if (row.properties != undefined) {
	                        value = _this.getPropertyValues(row.properties[label], row?.relatedLocalizedStrings, type=="display");
	                    } 
	                    if (!value && row[label] != undefined) {
	                        value = row[label];
	                    } 
	                    if (typeof value == "boolean") {
	                        value = __translator.translate("view.dialog." + value);
	                    }
	                    if (value) {
							if (value.length > 300 && type=="display") {
								return value.substring(0, 300) + '...';
							} else {
								return value;
							}
						}
	                    return "";
					}
					if (special) {
						return _this.renderSpecialColumn(special, row, type, val, meta);
					}
                },
            };
            
            columnDefs.push(def);
        });
        
        return columnDefs;
    }
        
    getPropertyValues(property, relatedLocalizedStrings, display) {
		if (property == undefined || property == null) {
            return '';
        }
		if (Array.isArray(property)) {
			return property.map(p => this.getPropertyValues(p, relatedLocalizedStrings, display)).join(", ");
		}
		if (typeof property === 'object') {
			if (Object.hasOwn(property, '@value')) {
				if (property.entryReference===false) {
					return '<span class="badge rounded-pill text-bg-primary">' + __translator.translate("view.model.freetext") + '</span> ' + property["@value"];	
				} else {
					return property["@value"];
				}
			} else if (Object.hasOwn(property, '@reference')) {
				return this.renderLocalizedDisplayString(property["@reference"], relatedLocalizedStrings, display);		
			} else {
				var hierarchicalProp = "";
				Object.keys(property).forEach(function(key,index) {
				    hierarchicalProp += property[key];
				    if (index < Object.keys(property).length - 1) {
						hierarchicalProp += " | ";
					}
				});
				return hierarchicalProp;
			}
		}
		return property;
    }
    
    renderLocalizedDisplayString(reference, relatedLocalizedStrings, display) {
		if (relatedLocalizedStrings) {
			for (const str of relatedLocalizedStrings) {
				if (str.entityId==reference && Object.hasOwn(str, '@value')) {
					if (display) {
						return '<span class="badge rounded-pill text-bg-secondary">' + __translator.translate("view.model.entity") + '</span>&nbsp;' + str["@value"]
					} else {
						return str["@value"];
					}
				}
			}
		}
		return reference;
	}
    
    renderValueColumn(label, row, type, val, meta) {
		return undefined;
	}
    
    renderSpecialColumn(special, row, type, val, meta) {
		return "";
	}
    
    addExportButtons(buttons) {
        if (this.options.includeExportButtons) {
            buttons.push({
                extend: 'collection',
                text: this.renderButtonTitle('export'),
                className: 'btn btn-outline-primary',
                buttons: ['copy', 'csv', 'print']
            });
        }
    }
    
    addColumnsButtons(buttons) {
        if (this.options.includeColumnReorder) {
            buttons.push({
                extend: 'collection',
                text: this.renderButtonTitle('columns'),
                className: 'btn btn-outline-primary',
                buttons: [
                    {
                        extend: 'colvis',
                        text: this.renderButtonTitle('column_visibility'),
                        collectionLayout: 'fixed columns',
                        collectionTitle: __translator.translate('view.data_tables.btn.column_visibility.control_message'),
                    }, {
                        extend: 'colvisRestore',
                        text: this.renderButtonTitle('restore_columns'),
                    }, {
                        text: this.renderButtonTitle('show_all_columns'),
                        className: 'btn btn-outline-primary',
                        // event, table, button, buttonConfig
                        action: function(e, dt, node, config) {
                            dt.columns().visible(true);
                        }
                    }, {
                        extend: "fixedColumns",
                        text: this.renderButtonTitle('fixed_columns'),
                        config: {
                            left: 2,
                        }
                    }
                ]
            });
        }
    }
    
    addSearchPaneButton(buttons) {
        const _this = this;
        if (this.options.includeSearchPane) {
            buttons.push({
                text: this.renderButtonTitle('filter'),
                className: 'btn btn-outline-secondary btn-search-pane',
                action: function() { _this.toggleSearchPaneVisibility(); }
            });
        }
    }
    
    cycleRefresh() {
        const _this = this;
        if (this.options.cyclicRefresh) {
            setTimeout(function() { _this.refresh(); _this.cycleRefresh(); }, _this.options.refreshInterval);
        };
    }
    
    toggleSearchPaneVisibility() {
        const searchPane = $(this.options.containerSelector + " .search-panes-container");
        if ($(searchPane).hasClass("d-none")) {
            $(searchPane).removeClass("d-none");
            this.table.searchPanes.resizePanes();
        } else {
            $(searchPane).addClass("d-none");
        }
    }
    
    assignTableEvents() {
        const _this = this;
        this.table.on('draw', function() {
            // Change filter button design depending on filter state
            if ($(".dtsp-searchPane .dtsp-selected").length > 0) {
                $(".btn-search-pane").removeClass("btn-outline-secondary").addClass("btn-secondary");
            } else {
                $(".btn-search-pane").removeClass("btn-secondary").addClass("btn-outline-secondary");
            }
        });

        // Click on row event
        if (this.options.clickRowEvent) {
	        this.table.on('click', 'tbody tr', function() {
	            const data = _this.table.row(this).data();
	            _this.handleRowClick(data);
	        });
        }
    }
    
    handleRowClick(data) {
		window.location = __util.composeUrl("entity/" + data.entityId + "/");
	}
    
    /* Just an 'abstract' method that is intended to be overridden */
    handleSelection(id) { }
    
    handleAjaxError(xhr, textStatus, error) {
        let errorCode = "N/A";
        if (xhr.responseJson?.status) {
            errorCode = xhr.responseJson.status;
        } else if (xhr.status) {
            errorCode = xhr.status;
        }
        bootbox.alert({
            title: __translator.translate("view.error.generic_ajax.head"),
            message: __translator.translate("view.error.generic_ajax.body_code", errorCode)
        });
        this.error = true;
        if (this.table.fnProcessingIndicator) {
            this.table.fnProcessingIndicator(false);
        }
    }
    
    refresh() {
        const searchPane = $(this.options.containerSelector + " .search-panes-container");

        // Do not reload while user is working on filters as they are visually impacted
        if (!this.error && this.table != null && ($(searchPane).length==0 || $(searchPane).hasClass("d-none"))) {
            this.table.ajax.reload(null, false);
        }
    }
    
    handleRefresh(oSettings) {
        // This is the case for the pre-load callback
        if (oSettings.aoData === null || oSettings.aoData === undefined ||
            (oSettings.aoData instanceof Array && oSettings.aoData.length == 0)) {
            return;
        }

        // We arrive here after data has been loaded into the table
        if (this.table != null) {
            this.table.$("[data-bs-toggle='tooltip']").tooltip({
                'delay': { show: 500, hide: 0 }
            });
        }
    }
    
    // Additional translations that are not covered by defaults
	getLanguage() {
		return {
			buttons: {
				copy: this.renderButtonTitle('copy'),
		        copyTitle: __translator.translate('view.data_tables.btn.copy.messageTitle'),
		        copySuccess: {
		            _: __translator.translate('view.data_tables.btn.copy.copySuccess.n'),
		            1: __translator.translate('view.data_tables.btn.copy.copySuccess.1'),
		        },
		        csv: this.renderButtonTitle('csv'),
		        print: this.renderButtonTitle('print'),
		    },
		    searchPanes: { 
				collapse: __translator.translate("view.data_tables.btn.filter"),
				title: {
		            _: __translator.translate('view.data_tables.search_panes.title._'),
		            0: __translator.translate('view.data_tables.search_panes.title.0'),
		            1: __translator.translate('view.data_tables.search_panes.title.1'),
		        },
		        count: __translator.translate('view.data_tables.search_panes.count'),
		    	countFiltered: __translator.translate('view.data_tables.search_panes.count_filtered'),
		    	
		    	collapseMessage: '<i class="bi bi-dash-square"></i> ' + __translator.translate('view.data_tables.search_panes.collapse_all'),
		    	showMessage: '<i class="bi bi-plus-square"></i> ' + __translator.translate('view.data_tables.search_panes.expand_all'),
		    	loadMessage: __translator.translate('view.data_tables.search_panes.load_message'),
		    	emptyPanes: __translator.translate('view.data_tables.search_panes.empty_panes'),
		    	emptyMessage: __translator.translate('view.data_tables.search_panes.empty_message'),
		    	clearMessage: '<i class="bi bi-x-square"></i> ' + __translator.translate('view.data_tables.search_panes.clear_all'),
			}
		}
	};

	renderButtonTitle(identifier, code, bsIcon) {
		let iIcon = "";
		if (this.options.useBootstrapIcons) {
			if (bsIcon) {
				iIcon = '<i class="bi ' + bsIcon + '"></i> '	
			} else if (this.options.actionIcons?.hasOwnProperty(identifier)) {
				iIcon = '<i class="bi ' + this.options.actionIcons[identifier] + '"></i> '
			}
		}
	
		let label;
		if (code==undefined) {
			label = 'view.data_tables.btn.' + identifier;
		} else if (code!=null && code!="") {
			label = code;
		}	
		
		return iIcon + (label==undefined ? "" : __translator.translate(label));
	}
}
