package de.uniba.minf.registry.view.model;

import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.util.List;

import de.uniba.minf.registry.model.Property;
import de.uniba.minf.registry.model.PropertyValue;
import de.uniba.minf.registry.model.PropertyValueList;
import de.uniba.minf.registry.model.definition.PropertyDefinition;
import lombok.Data;
import lombok.RequiredArgsConstructor;

@Data
@RequiredArgsConstructor
public abstract class BasePropertyViewItem implements PropertyViewItem {
	private final PropertyDefinition definition;
	
	private boolean readonly;
	
	@Override public boolean isSimple() { return definition.isSimple(); }
	@Override public boolean isVocabulary() { return definition.isVocabulary(); }
	@Override public boolean isHierarchical() { return definition.isHierarchical(); }
	
	public PropertyValue getValue() {
		if (isSimple()) {
			return this.getPropertyValue(SimplePropertyViewItem.class.cast(this).getProperty());
		} else if (isVocabulary()) {
			return this.getPropertyValue(VocabularyPropertyViewItem.class.cast(this).getProperty());
		} else {
			return null;
		}
	}
	
	public boolean isMultiValue() {
		PropertyValue value = this.getValue();
		return value!=null && PropertyValueList.class.isAssignableFrom(value.getClass()); 
	}
	
	private PropertyValue getPropertyValue(Property entity) {
		return entity==null ? null : entity.getValue();
	}
	
	public String getUrlSafeLabel() {
		return URLEncoder.encode(definition.getName(), StandardCharsets.UTF_8);
	}
	
	@Override
	public boolean allowsMultiplicity() {
		return this.getDefinition().isMultiple() || this.getDefinition().isMultilingual();
	}
	
	@Override
	public boolean isValueList() {
		return this.allowsMultiplicity();
	}
	
	@Override
	public List<PropertyValue> valuesAsList(boolean includeEmptyPlaceholder) {
		return this.valuesAsList();
	}
}
