package de.uniba.minf.registry.view.helper;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import de.uniba.minf.registry.model.Property;
import de.uniba.minf.registry.model.PropertyList;
import de.uniba.minf.registry.model.definition.EntityDefinition;
import de.uniba.minf.registry.model.definition.HierarchicalPropertyDefinition;
import de.uniba.minf.registry.model.definition.PropertyDefinition;
import de.uniba.minf.registry.model.definition.PropertyDefinitionBlock;
import de.uniba.minf.registry.model.definition.VocabularyPropertyDefinition;
import de.uniba.minf.registry.model.vocabulary.VocabularyDefinition;
import de.uniba.minf.registry.model.vocabulary.VocabularyEntry;
import de.uniba.minf.registry.repository.EntityDefinitionRepository;
import de.uniba.minf.registry.repository.EntityRepository;
import de.uniba.minf.registry.repository.VocabularyDefinitionRepository;
import de.uniba.minf.registry.repository.VocabularyEntryRepository;
import de.uniba.minf.registry.view.model.EntityViewItem;
import de.uniba.minf.registry.view.model.HierarchicalPropertyViewItem;
import de.uniba.minf.registry.view.model.PropertyBlockViewItem;
import de.uniba.minf.registry.view.model.PropertyViewItem;
import de.uniba.minf.registry.view.model.SimplePropertyViewItem;
import de.uniba.minf.registry.view.model.VocabularyPropertyViewItem;

@Component
public class PropertyViewItemCombiner {
	
	private static final String INTERNAL_VOCABULARY_IDENTIFIER_ENTITIES = "_entities";
	
	@Autowired private VocabularyDefinitionRepository vocabularyDefinitionRepository;
	@Autowired private VocabularyEntryRepository vocabularyEntryRepository;
	
	@Autowired private EntityRepository entityRepository;
	@Autowired private EntityDefinitionRepository entityDefinitionRepository;
		
	public List<PropertyBlockViewItem> getFromDefinitionBlocksAndEntity(List<PropertyDefinitionBlock> definitionBlocks, List<Property> values) {
		if (definitionBlocks==null) {
			return new ArrayList<>(0);
		}
		List<PropertyBlockViewItem> blockViewItems = new ArrayList<>();
		
		Collection<EntityDefinition> entityDefinitions = entityDefinitionRepository.findAllLatest();
		
		PropertyBlockViewItem blockViewItem;
		for (PropertyDefinitionBlock definitionBlock : definitionBlocks) {
			blockViewItem = new PropertyBlockViewItem();
			blockViewItem.setPropertyViewItems(this.getFromDefinitionAndEntity(definitionBlock.getProperties(), entityDefinitions, values));
			blockViewItem.setName(definitionBlock.getName());
			blockViewItem.setIdentifier(definitionBlock.getIdentifier());
			blockViewItem.setMessageCode(definitionBlock.getMessageCode());
			blockViewItems.add(blockViewItem);
		}
		return blockViewItems;
	}
	
	public List<PropertyViewItem> getFromDefinitionAndEntity(List<PropertyDefinition> definitions, List<Property> values) {
		Collection<EntityDefinition> entityDefinitions = entityDefinitionRepository.findAllLatest();
		return this.getFromDefinitionAndEntity(definitions, entityDefinitions, values);
	}
	
	public PropertyViewItem getFromDefinitionAndEntity(PropertyDefinition definition, Property value) {
		Collection<EntityDefinition> entityDefinitions = entityDefinitionRepository.findAllLatest();
		return this.getFromDefinitionAndEntity(definition, entityDefinitions, value);
	}
	
	private List<PropertyViewItem> getFromDefinitionAndEntity(List<PropertyDefinition> definitions, Collection<EntityDefinition> entityDefinitions, List<Property> values) {
		if (definitions==null) {
			return new ArrayList<>(0);
		}
		
		List<VocabularyDefinition> vocabularyDefinitions = vocabularyDefinitionRepository.findAll();
		Map<String, List<VocabularyEntry>> vocabularyEntryMap = new HashMap<>();
		
		return this.getFromDefinitionAndEntityWithVocabularies(definitions, values, vocabularyDefinitions, entityDefinitions, vocabularyEntryMap);
	}
	
	private PropertyViewItem getFromDefinitionAndEntity(PropertyDefinition definition, Collection<EntityDefinition> entityDefinitions, Property value) {
		List<VocabularyDefinition> vocabularyDefinitions = vocabularyDefinitionRepository.findAll();
		Map<String, List<VocabularyEntry>> vocabularyEntryMap = new HashMap<>();
		
		return this.getFromDefinitionAndEntityWithVocabularies(definition, value, vocabularyDefinitions, entityDefinitions, vocabularyEntryMap);
	}
	
	private List<PropertyViewItem> getFromDefinitionAndEntityWithVocabularies(List<PropertyDefinition> definitions, List<Property> values, List<VocabularyDefinition> vocabularyDefinitions, Collection<EntityDefinition> entityDefinitions, Map<String, List<VocabularyEntry>> vocabularyEntryMap) {
		List<PropertyViewItem> viewItems = new ArrayList<>(definitions.size());
		Property value;
		for (PropertyDefinition definition : definitions) {			
			value = values==null ? null : values.stream().filter(v -> v.getLabel().equals(definition.getName())).findFirst().orElse(null);
			viewItems.add(this.getFromDefinitionAndEntityWithVocabularies(definition, value, vocabularyDefinitions, entityDefinitions, vocabularyEntryMap));
		}
		return viewItems;
	}
	
	private PropertyViewItem getFromDefinitionAndEntityWithVocabularies(PropertyDefinition definition, Property value, List<VocabularyDefinition> vocabularyDefinitions, Collection<EntityDefinition> entityDefinitions, Map<String, List<VocabularyEntry>> vocabularyEntryMap) {
		if (VocabularyPropertyDefinition.class.isAssignableFrom(definition.getClass())) {
			VocabularyPropertyViewItem vocabularyViewItem = new VocabularyPropertyViewItem(definition, value);
			this.processVocabularyDefinition(vocabularyViewItem, VocabularyPropertyDefinition.class.cast(definition).getVocabulary(), vocabularyDefinitions, entityDefinitions, vocabularyEntryMap, null);
			return vocabularyViewItem;
		} else if (HierarchicalPropertyDefinition.class.isAssignableFrom(definition.getClass())) {
			HierarchicalPropertyViewItem hierarchicalViewItem = new HierarchicalPropertyViewItem(definition);
			this.processHierarchicalDefinition(hierarchicalViewItem, HierarchicalPropertyDefinition.class.cast(definition), value, vocabularyDefinitions, entityDefinitions, vocabularyEntryMap);
			return hierarchicalViewItem;
		} else if (definition.isMultilingual()) {
			List<String> allowedLanguages = definition.getAllowedLanguages();
			VocabularyPropertyDefinition langPropertyDefintion = new VocabularyPropertyDefinition();
			langPropertyDefintion.setQuery(allowedLanguages.isEmpty());
			langPropertyDefintion.setVocabulary("_language");
			langPropertyDefintion.setStrict(true);
			langPropertyDefintion.setIdentifier(definition.getIdentifier() + "#lang");
			
			VocabularyPropertyViewItem langVocabularyViewItem = new VocabularyPropertyViewItem(langPropertyDefintion, value);
			this.processVocabularyDefinition(langVocabularyViewItem, langPropertyDefintion.getVocabulary(), vocabularyDefinitions, entityDefinitions, vocabularyEntryMap, allowedLanguages);
			return new SimplePropertyViewItem(definition, value, langVocabularyViewItem);
		} else {
			return new SimplePropertyViewItem(definition, value);
		}
	}
	
	private void processVocabularyDefinition(VocabularyPropertyViewItem viewItem, String vocabulary, List<VocabularyDefinition> vocabularyDefinitions, Collection<EntityDefinition> entityDefinitions, Map<String, List<VocabularyEntry>> vocabularyEntryMap, List<String> allowedVocabularyKeys) {
		if (vocabulary==null) {
			return;
		}
		
		// Special internal vocabulary of all entity types
		// Do we need this in a more central place or is this view-relevant only?
		if (viewItem.getDefinition().asVocabulary().isEntity() && viewItem.getProperty()!=null) {
			List<EntityViewItem> entityViewItems = new ArrayList<>();
			entityRepository.findLatestByEntityIds(viewItem.getProperty().valuesAsList().stream().map(v -> v.asText()).toList()).stream().forEach(e -> {
				EntityViewItem vi = new EntityViewItem();
				vi.setEntityId(e.getEntityId());
				vi.setUniqueId(e.getUniqueId());
				
				Optional<EntityDefinition> ed = entityDefinitions.stream().filter(def -> def.getName().equals(e.getDefinitionName())).findFirst();
				if (ed.isPresent()) {
					vi.setPreviewProperties(PropertyPreviewHelper.get().extractSubproperties(ed.get().getPreviewProperties(), e.getProperties()));
				}
				entityViewItems.add(vi);
			});
			viewItem.setEntityViewItems(entityViewItems);
		} else if (vocabulary.equals(INTERNAL_VOCABULARY_IDENTIFIER_ENTITIES)) {
			VocabularyDefinition internalVocabularyDefinition = new VocabularyDefinition();
			internalVocabularyDefinition.setName(INTERNAL_VOCABULARY_IDENTIFIER_ENTITIES);
			viewItem.setVocabularyDefinition(internalVocabularyDefinition);
			viewItem.setVocabularyEntries(entityDefinitionRepository.findAllLatest().stream().filter(ed -> !ed.isSystem()).map(ed -> { 
				VocabularyEntry ve = new VocabularyEntry();
				ve.setKey(ed.getName());
				ve.setPrimaryValue(ed.getLabel());
				return ve;
			}).toList());
			vocabularyEntryMap.put(INTERNAL_VOCABULARY_IDENTIFIER_ENTITIES, viewItem.getVocabularyEntries());
		} else {
			Optional<VocabularyDefinition> vocabularyDefinition = vocabularyDefinitions.stream().filter(vd -> vd.getName().equals(vocabulary)).findFirst();
			if (vocabularyDefinition.isPresent()) {
				viewItem.setVocabularyDefinition(vocabularyDefinition.get());
				if (vocabularyEntryMap.containsKey(vocabularyDefinition.get().getName())) {
					viewItem.setVocabularyEntries(getVocabularyEntries(vocabularyEntryMap.get(vocabularyDefinition.get().getName()), allowedVocabularyKeys));
				} else {
					viewItem.setVocabularyEntries(getVocabularyEntries(vocabularyEntryRepository.findByDefinition(vocabularyDefinition.get().getName()), allowedVocabularyKeys));	
				}
				vocabularyEntryMap.put(vocabularyDefinition.get().getName(), viewItem.getVocabularyEntries());
			}
		}
	}
	
	private List<VocabularyEntry> getVocabularyEntries(List<VocabularyEntry> availableEntries, List<String> allowedVocabularyKeys) {
		if (availableEntries==null || allowedVocabularyKeys==null || allowedVocabularyKeys.isEmpty()) {
			return availableEntries;
		} else {
			return availableEntries.stream().filter(ve -> allowedVocabularyKeys.contains(ve.getKey())).toList();
		}
	}
	
	private void processHierarchicalDefinition(HierarchicalPropertyViewItem viewItem, HierarchicalPropertyDefinition definition, Property value, List<VocabularyDefinition> vocabularyDefinitions, Collection<EntityDefinition> entityDefinitions, Map<String, List<VocabularyEntry>> vocabularyEntryMap) {
		List<List<PropertyViewItem>> subitemList = new ArrayList<>();
		List<PropertyViewItem> subitems;
		
		List<List<PropertyViewItem>> previewitemList = new ArrayList<>();
		List<PropertyViewItem> previewitems;
		
		List<PropertyDefinition> previewProperties = definition.getPreviewProperties();
		
		if (value!=null && value.getProperties()!=null) {
			for (PropertyList propertyList : value.getProperties()) {
				subitems = this.getFromDefinitionAndEntityWithVocabularies(definition.getProperties(), propertyList.getProperties(), vocabularyDefinitions, entityDefinitions, vocabularyEntryMap);
				subitemList.add(subitems);
				
				previewitems = subitems.stream().filter(s -> previewProperties.contains(s.getDefinition())).toList();
				previewitemList.add(previewitems);
			}
		}
		viewItem.setSubitems(subitemList);
		viewItem.setPreviewItems(previewitemList);
	}
}
