package de.uniba.minf.registry.service;

import java.net.URL;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.Set;

import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpMethod;
import org.springframework.http.ResponseEntity;
import org.springframework.http.client.ClientHttpRequestFactory;
import org.springframework.http.client.HttpComponentsClientHttpRequestFactory;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonInclude.Include;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import de.uniba.minf.registry.model.vocabulary.VocabularyDefinition;
import de.uniba.minf.registry.model.vocabulary.VocabularyEntry;
import de.uniba.minf.registry.model.vocabulary.VocabularyLookupException;
import de.uniba.minf.registry.model.vocabulary.VocabularyLookupService;
import de.uniba.minf.registry.repository.VocabularyDefinitionRepository;
import de.uniba.minf.registry.repository.VocabularyEntryRepository;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;

@Slf4j
@Component
public class VocabularyLookupServiceImpl implements VocabularyLookupService, InitializingBean {

	private static final int HTTP_TIMEOUT = 5000;
	
	@Autowired private VocabularyDefinitionRepository vocabularyDefinitionRepository;
	@Autowired private VocabularyEntryRepository vocabularyEntryRepository;
	
	@Autowired @Qualifier("yamlMapper") private ObjectMapper yamlMapper;
	@Autowired @Qualifier("jsonMapper") private ObjectMapper jsonMapper;
	
	@Value("#{${debug}==true || ${debugging.onlinelookups}==true}")
	private boolean debugLookups;
	
	@Override
	public void afterPropertiesSet() throws Exception {
		if (debugLookups) {
			log.debug("Debugging mode for vocabulary lookups configured");
		}
	}
	
	@Data
	@JsonInclude(Include.NON_NULL)
	class Query {
		private final String query;
		private final String id;
	}
	
	@Override
	public boolean vocabularyAvailable(String vocabulary) throws VocabularyLookupException {
		VocabularyDefinition definition;
		try {
			definition = this.getAndCheckVocabulary(vocabulary);
		} catch (VocabularyLookupException ex) {
			return false;
		}
		
		/* This merely tests if the server responds at the provided url
		 * TODO: more specific, functionality based checks could be set up here */
		RestTemplate restTemplate = new RestTemplate(getClientHttpRequestFactory());
		Set<HttpMethod> optionsForAllow = restTemplate.optionsForAllow(definition.getEndpointUrl());
		return optionsForAllow.contains(HttpMethod.valueOf(definition.getEndpointMethod()==null ? "GET" : definition.getEndpointMethod()));
	}
	
	@Override
	public boolean canResolveId(String vocabulary, String id) throws VocabularyLookupException {
		VocabularyDefinition definition;
		try {
			definition = this.getAndCheckVocabulary(vocabulary);
		} catch (VocabularyLookupException ex) {
			return false;
		}
		
		// Vocabulary entry already used and saved in database
		if (this.find(vocabulary, id).isPresent()) {
			return true;
		}
		if (definition.isRemote()) {
			return !fetch(vocabulary, definition.getEndpointUrl(), definition.getEndpointMethod(), id, null).isEmpty();
		}
		return false;
	}
	
	@Override
	public VocabularyEntry resolve(String vocabulary, String id) throws VocabularyLookupException {
		VocabularyDefinition definition;
		try {
			definition = this.getAndCheckVocabulary(vocabulary);
		} catch (VocabularyLookupException ex) {
			return null;
		}
		
		Optional<VocabularyEntry> ve = this.find(vocabulary, id);
		// Vocabulary entry already used and saved in database
		if (ve.isPresent()) {
			return ve.get();
		}
		if (definition.isRemote()) {
			List<VocabularyEntry> veList = this.fetch(vocabulary, definition.getEndpointUrl(), definition.getEndpointMethod(), id, null);
			if (veList.isEmpty()) {
				return null;
			}		
			return veList.get(0);
		}
		return null;
	}
	
	@Override
	public List<VocabularyEntry> search(String vocabulary, String query) throws VocabularyLookupException {
		VocabularyDefinition definition;
		try {
			definition = this.getAndCheckVocabulary(vocabulary);
		} catch (VocabularyLookupException ex) {
			return new ArrayList<>(0);
		}
		
		List<VocabularyEntry> result;
		Optional<VocabularyEntry> ve = this.find(vocabulary, query);
		if (ve.isPresent()) {
			result = new ArrayList<>();
			result.add(ve.get());
		} else if (definition.isRemote()) {
			// TODO: This should no longer be required, rework as soon an GS is live again
			//result = this.fetch(vocabulary, definition.getEndpointUrl(), definition.getEndpointMethod(), query, null);
			result = new ArrayList<>();
		} else {
			result = vocabularyEntryRepository.findByDefinitionAndQuery(vocabulary, query);
		}
		
		if (result.isEmpty() && definition.isRemote()) {
			result = this.fetch(vocabulary, definition.getEndpointUrl(), definition.getEndpointMethod(), null, query);
		}
		return result;
	}
	
	
	private VocabularyEntry processItem(String vocabulary, JsonNode node) throws VocabularyLookupException {
		VocabularyEntry ve;
		try {
			if (node.isObject() && node.has("vocabularyEntry")) {
				ve = jsonMapper.treeToValue(node.get("vocabularyEntry"), VocabularyEntry.class);
				ve.setDefinitionName(vocabulary);
				
				if (this.find(vocabulary, ve.getKey()).isEmpty()) {
					log.debug("Persisting fetched entry with key '{}' to vocabulary '{}'", vocabulary, ve.getKey());
					vocabularyEntryRepository.save(ve);
				}
				return ve;
			} 
		} catch (Exception e) {
			throw new VocabularyLookupException(vocabulary, null, false, "Failed to process vocabulary endpoint response", e);
		}
		
		return null;
	}
	
	private Optional<VocabularyEntry> find(String vocabulary, String id) throws VocabularyLookupException {
		Optional<VocabularyEntry> ve = vocabularyEntryRepository.findByDefinitionAndKey(vocabulary, id); 
		if (this.debugLookups) {
			log.debug("Local query on vocabulary '{}' with id '{}' returned {} entry", vocabulary, id, ve.isEmpty() ? "NO" : "matching");
		}
		return vocabularyEntryRepository.findByDefinitionAndKey(vocabulary, id);
	}
	
	private List<VocabularyEntry> fetch(String vocabulary, String url, String method, String id, String query) throws VocabularyLookupException {
		RestTemplate restTemplate = new RestTemplate(getClientHttpRequestFactory());
		try {
			HttpMethod httpMethod = HttpMethod.valueOf(method==null ? "GET" : method);
			
			HttpEntity<Query> request = new HttpEntity<>(new Query(query, id));
			ResponseEntity<String> response = restTemplate.exchange(url, httpMethod, request, String.class);
			
			List<VocabularyEntry> result = new ArrayList<>();
					
			JsonNode root = jsonMapper.readTree(response.getBody());
			JsonNode items = root.get("response").get("items");
			
			final VocabularyEntry ve = this.processItem(vocabulary, items);			
			if (ve!=null && ve.getKey()!=null && ve.getPrimaryValue()!=null && result.stream().noneMatch(r->r.getKey().equals(ve.getKey()))) {
				result.add(ve);
			} else if (items.isArray()) {
				for (JsonNode item : items) {
					final VocabularyEntry veArr = this.processItem(vocabulary, item);
					if (veArr!=null && veArr.getKey()!=null && veArr.getPrimaryValue()!=null && result.stream().noneMatch(r->r.getKey().equals(veArr.getKey()))) {
						result.add(veArr);
					}
				}
			}
			if (this.debugLookups) {
				log.debug("Remote query on vocabulary '{}' with {}{} returned {} entry", vocabulary, id!=null ? ("id '" + id + "'") : "", query!=null ? ("query '" + query + "'") : "", result.isEmpty() ? "NO" : result.size() + " matching");
			}
			return result;
		} catch (VocabularyLookupException e) {
			throw e;
		} catch (Exception e) {
			throw new VocabularyLookupException(vocabulary, String.format("id: %s; query: %s", id, query), true, "Failed to fetch vocabulary entries", e);
		} 
	}
	
	private VocabularyDefinition getAndCheckVocabulary(String vocabulary) throws VocabularyLookupException {
		VocabularyDefinition definition = vocabularyDefinitionRepository.findCurrentByName(vocabulary); 
		if (definition==null) {
			throw new VocabularyLookupException(vocabulary, null, false, "Vocabulary definition not available");
		}
		if (definition.isRemote()) {
			if (definition.getEndpointUrl()==null) {
				throw new VocabularyLookupException(vocabulary, null, false, "Endpoint for vocabulary not specified");
			}
			try {
				new URL(definition.getEndpointUrl()).toURI();
				return definition;
		    } catch (Exception e) {
		    	throw new VocabularyLookupException(vocabulary, null, false, "Endpoint URL invalid for vocabulary", e);
		    }
		} else {
			return definition;
		}
	}
	
	private ClientHttpRequestFactory getClientHttpRequestFactory() {
	    HttpComponentsClientHttpRequestFactory clientHttpRequestFactory = new HttpComponentsClientHttpRequestFactory();
	    clientHttpRequestFactory.setConnectTimeout(HTTP_TIMEOUT);
	    return clientHttpRequestFactory;
	}
}
