package de.uniba.minf.registry.migration;

import static de.uniba.minf.registry.model.serialization.base.BaseDefinitionSerializationConstants.ENTITY_NAME_FIELD;
import static de.uniba.minf.registry.model.serialization.base.BaseDefinitionSerializationConstants.VOCABULARY_FIELD;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.List;

import org.apache.commons.io.FileUtils;
import org.apache.tika.Tika;
import org.apache.tika.io.TikaInputStream;
import org.apache.tika.metadata.Metadata;
import org.apache.tika.metadata.TikaCoreProperties;
import org.joda.time.DateTime;
import org.joda.time.format.DateTimeFormat;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.Resource;
import org.springframework.core.io.ResourceLoader;
import org.springframework.core.io.support.ResourcePatternUtils;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.stereotype.Component;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;

import de.uniba.minf.registry.model.VersionInfo;
import de.uniba.minf.registry.model.definition.EntityDefinition;
import de.uniba.minf.registry.model.entity.Entity;
import de.uniba.minf.registry.model.vocabulary.VocabularyDefinition;
import de.uniba.minf.registry.model.vocabulary.VocabularyEntry;
import de.uniba.minf.registry.repository.VersionInfoRepository;
import de.unibamberg.minf.core.web.exception.ApplicationSetupException;
import lombok.extern.slf4j.Slf4j;

@Slf4j
@Component
public class MigrationServiceImpl implements InitializingBean {
	private static final String VERSION_HASH_PREFIX = "Registry";
	
	@Value("${paths.backups:#{null}}")
	private String backupsPath;
	
	@Value("${paths.definitions:#{null}}")
	private String customDefinitions;
	
	@Value("${paths.entries:#{null}}")
	private String customEntries;
	
	@Autowired ResourceLoader resourceLoader;
	@Autowired @Qualifier("yamlMapper") private ObjectMapper yamlMapper;
	@Autowired @Qualifier("jsonMapper") private ObjectMapper jsonMapper;

	@Autowired private Tika tika;
	
	@Autowired private MongoTemplate mongoTemplate;
	
	@Autowired private VersionInfoRepository versionInfoRepository;
		
	private final MessageDigest md;
	
	public MigrationServiceImpl() throws NoSuchAlgorithmException {
		md = MessageDigest.getInstance("MD5");
	}
	
	@Override
	public void afterPropertiesSet() throws Exception {
		List<String> versions = new ArrayList<>();
		List<VersionInfo> versionInfos = versionInfoRepository.findAll();
		for (VersionInfo vi : versionInfos) {
			if (!vi.getVersionHash().equals(new String(md.digest(new String(VERSION_HASH_PREFIX + vi.getVersion()).getBytes(StandardCharsets.UTF_8)), StandardCharsets.UTF_8))) {
				log.error("Cancelling migration checks: failed to compare version hashes. Is the correct database configured?");
				return;
			}
			versions.add(vi.getVersion());
		}
		this.performMigrations(versions);
	}
	
	private void performMigrations(List<String> existingVersions) throws Exception {
		boolean backedUp = false;
		
		backedUp = backedUp | this.migrate(existingVersions, !backedUp, "5.0", this::importSystemDefinitionsAndEntriesForVersion);
		
		// TODO: Not yet for release
		// backedUp = backedUp | this.migrate(existingVersions, !backedUp, "5.1", this::importSystemDefinitionsAndEntriesForVersion);
		
		// If there are files at configured paths -> backup and import without VersionInfo
		backedUp = backedUp | this.importConfiguredDefinitionsAndEntries();
		
		log.info("Migration performed: {}", backedUp);
	}
	
	
	private boolean importConfiguredDefinitionsAndEntries() {
		try {
			File f;
			if (customDefinitions!=null) {
				f = new File(customDefinitions);
				if (f.exists()) {
					this.scanHandleAndDeleteFiles(f, true);
				}
			}
			if (customEntries!=null) {
				f = new File(customEntries);
				if (f.exists()) {
					this.scanHandleAndDeleteFiles(f, false);
				}
			}
		} catch (IOException | ApplicationSetupException e) {
			log.error("Failed to import configured system definitions and entries", e);
			return false;
		}
		return true;
	}
	private boolean importSystemDefinitionsAndEntriesForVersion(String version) {
		try {
			Resource[] resources = ResourcePatternUtils.getResourcePatternResolver(resourceLoader).getResources("classpath:system_definitions/" + version + "*");
			for (Resource resource : resources) {
				this.handleResource(resource, true);
			}
			
			resources = ResourcePatternUtils.getResourcePatternResolver(resourceLoader).getResources("classpath:system_entries/" + version + "*");
			for (Resource resource : resources) {
				this.handleResource(resource, false);
			}
		} catch (IOException | ApplicationSetupException ex) {
			log.error("Failed to import system definitions and entries", ex);
			return false;
		}
		return true;
	}
	
	private void scanHandleAndDeleteFiles(File f, boolean definitions) throws IOException, ApplicationSetupException {
		if (f.isDirectory()) {
			for (File file : f.listFiles()) {
				this.handleResource(resourceLoader.getResource("file:" + file.getAbsolutePath()), definitions);
				FileUtils.delete(file);
			}
		} else {
			this.handleResource(resourceLoader.getResource("file:" + f.getAbsolutePath()), definitions);
			FileUtils.delete(f);
		}
	}
		
	private void handleResource(Resource r, boolean definitions) throws IOException, ApplicationSetupException {
		String mimeType = this.detectMimeType(r);
		log.info("Importing {} from {} ({})", definitions ? "definition" : "entries", r.getFilename(), mimeType);
		
		ObjectMapper mapper = this.getMapperForMimeType(mimeType);
		try (InputStream stream = r.getInputStream()) {
			JsonNode n = mapper.readTree(stream);
			if (n.isArray()) {
				for (JsonNode entry : n) {
					this.handleNode(mapper, entry, definitions);
				}
			} else {
				this.handleNode(mapper, n, definitions);
			}
		}
	}
	
	private String detectMimeType(Resource r) throws IOException {
		 try (TikaInputStream stream = TikaInputStream.get(r.getInputStream())) {
			 final Metadata metadata = new Metadata();
			 metadata.set(TikaCoreProperties.RESOURCE_NAME_KEY, r.getFilename());
			 return tika.detect(stream, metadata);
		 }
	}
	
	private ObjectMapper getMapperForMimeType(String mimeType) throws ApplicationSetupException {
		if (mimeType!=null && (mimeType.endsWith("/x-yaml") || mimeType.endsWith("/yaml")) ) {
			return yamlMapper;
		} else if (mimeType!=null && mimeType.endsWith("/json")) {
			return jsonMapper;
		} else {
			throw new ApplicationSetupException("Failed to initialize definitions or entries: unsupported MIME type: " + (mimeType==null ? "NULL" : mimeType), this.getClass());
		}
	}
	
	private void handleNode(ObjectMapper mapper, JsonNode n, boolean definitions) throws JsonProcessingException, IllegalArgumentException {
		if (definitions && n.has(ENTITY_NAME_FIELD)) {
			mongoTemplate.save(mapper.treeToValue(n, EntityDefinition.class));
		} else if (definitions && n.has(VOCABULARY_FIELD)) {
			mongoTemplate.save(mapper.treeToValue(n, VocabularyDefinition.class));
		} else if (!definitions && n.has("_" + VOCABULARY_FIELD)) {
			// SaveOrUpdate mechanism for vocabulary entries
			VocabularyEntry ve = mapper.treeToValue(n, VocabularyEntry.class);
			Query q = new Query();
			q.addCriteria(Criteria.where("key").is(ve.getKey()).and("definitionName").is(ve.getDefinitionName()));
			
			VocabularyEntry eExist = mongoTemplate.findOne(q, VocabularyEntry.class);
			if (eExist!=null) {
				ve.setUniqueId(eExist.getUniqueId());			
			}
			mongoTemplate.save(ve);
		} else {
			mongoTemplate.save(mapper.treeToValue(n, Entity.class));
		}
	}
	
	private boolean migrate(List<String> existingVersions, boolean backup, String version, MigrationAction migration) throws Exception {
		if (!existingVersions.contains(version)) {
			log.info("Migrating to version [{}]", version);	
			if (backup) {
				this.backupDb();
			}
			boolean success = migration.migrate(version);
			this.saveVersionInfo(version, success);
			log.info("Migration to version [{}] performed {}", version, success ? "sucessfully" : "WITH ERRORS");		
			return true;
		}
		return false;
	}
	
	private void backupDb() throws Exception {
		if (backupsPath==null) {
			log.warn("No path for backups (paths.backups) configured; skipping backup creation before applying migration");
			return;
		}
		String backupPath = backupsPath + File.separator + DateTime.now().toString(DateTimeFormat.forPattern("yyyyMMdd_HHmmss"));
		Files.createDirectories(Paths.get(new File(backupPath).toURI()));
		try {
			Runtime.getRuntime().exec(String.format("mongodump --out %s --db %s", backupPath, mongoTemplate.getDb().getName()));
			log.info("Backed up database {} to [{}]", mongoTemplate.getDb().getName(), backupPath);
		} catch (Exception e) {
			log.error("Failed to create mongodb backup", e);
			throw e;
		}
	}
	
	private void saveVersionInfo(String version, boolean success) {
		VersionInfo vi = new VersionInfo();
		vi.setUpdateWithErrors(!success);
		vi.setVersion(version);		
		vi.setVersionHash(new String(md.digest(new String(VERSION_HASH_PREFIX + vi.getVersion()).getBytes(StandardCharsets.UTF_8)), StandardCharsets.UTF_8));
		
		versionInfoRepository.save(vi);
	}
}
