package de.uniba.minf.registry.importer;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import de.uniba.minf.auth.spring.mvc.AuthInfoHelper;
import de.uniba.minf.core.rest.model.RestActionResponse.ApiActionStatus;
import de.uniba.minf.registry.model.Import;
import de.uniba.minf.registry.model.ImportResult;
import de.uniba.minf.registry.model.entity.Entity;
import de.uniba.minf.registry.repository.EntityRepository;
import de.uniba.minf.registry.repository.ImportRepository;
import de.uniba.minf.registry.service.EntityService;
import lombok.extern.slf4j.Slf4j;

@Slf4j
@Component
public class ImportServiceImpl implements ImportService, InitializingBean, PropertyChangeListener {

	@Autowired private EntityService entityService;
	@Autowired private EntityRepository entityRepo;
	@Autowired private ImportRepository importRepo;
	
	@Autowired private AuthInfoHelper authinfoHelper;
	
	protected ExecutorService importExecutor;
	private int maxPoolSize = 5;
	
	private Map<String, ImportResult> importRunnerIdResultMap;
	
	@Override
	public void afterPropertiesSet() throws Exception {
		importExecutor = Executors.newFixedThreadPool(maxPoolSize);
		importRunnerIdResultMap = new HashMap<>();
	}
	
	@Override
	public long deleteImport(String uniqueId) {	
		Optional<Import> i = importRepo.findById(uniqueId);
		if (i.isEmpty()) {
			return 0;
		}
		return this.deleteImport(i.get());
	} 
	
	@Override
	public long deleteImport(Import i) {	
		long count;
		
		Collection<Entity> entities = entityRepo.findByImportId(i.getUniqueId());
		count = entities.size();
		entityRepo.deleteAll(entities);
		log.info("Deleting read-only import, deleted {} associated entites", count);
		
		importRepo.delete(i);
		log.info("Deleted import: {}", i.getUniqueId());
		
		return count;
	}

	@Override
	public ApiActionStatus getStatus(String runnerId) {
		if (runnerId==null || !this.importRunnerIdResultMap.containsKey(runnerId)) {
			return ApiActionStatus.NONE;
		}
		ImportResult result = this.importRunnerIdResultMap.get(runnerId);
		if (result==null) {
			return ApiActionStatus.BUSY;
		}
		return ApiActionStatus.DONE;
	} 

	@Override
	public ImportResult getImportResult(String runnerId) {
		if (runnerId==null || !this.importRunnerIdResultMap.containsKey(runnerId)) {
			return null;
		}
		return this.importRunnerIdResultMap.get(runnerId);
	}
	
	@Override
	public void importAsync(ImportRunner ir) {
		
		this.importRunnerIdResultMap.put(ir.getImportRunnerId(), null);
		
		ir.addPropertyChangeListener(this);
		
		this.importExecutor.execute(ir);
	}

	@Override
	public void propertyChange(PropertyChangeEvent arg0) {
		ImportRunner ir = ImportRunner.class.cast(arg0.getSource());
		ImportResult result = arg0.getNewValue()==null ? null : ImportResult.class.cast(arg0.getNewValue());

		this.importRunnerIdResultMap.put(ir.getImportRunnerId(), result);
	}

	
}
