package de.uniba.minf.registry.importer;

import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.time.Instant;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.MessageSource;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import am.ik.yavi.core.ConstraintViolations;
import de.uniba.minf.registry.model.Import;
import de.uniba.minf.registry.model.ImportResult;
import de.uniba.minf.registry.model.definition.EntityDefinition;
import de.uniba.minf.registry.model.entity.Entity;
import de.uniba.minf.registry.model.validation.exception.ValidationConfigurationException;
import de.uniba.minf.registry.repository.ImportRepository;
import de.uniba.minf.registry.repository.ImportResultRepository;
import de.uniba.minf.registry.service.EntityDefinitionService;
import de.uniba.minf.registry.service.EntityService;
import de.uniba.minf.registry.view.helper.EntityVocabularyHelper;
import de.uniba.minf.registry.view.helper.PropertyDefinitionHelper;
import de.uniba.minf.registry.view.helper.ResolvedVocabularyEntry;
import de.uniba.minf.registry.view.helper.PropertyDefinitionHelper.UnknownProperty;
import de.uniba.minf.registry.view.model.ValidationViolationMessage;
import groovyjarjarantlr4.v4.runtime.misc.Args;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;

@Data
@Slf4j
@Component
@Scope("prototype")
public class ImportRunner implements Runnable {

	@Autowired private EntityService entityService;
	@Autowired private ImportService importService;
	
	@Autowired private EntityDefinitionService entityDefService;
	@Autowired private PropertyDefinitionHelper propertyDefinitionHelper;
	@Autowired private EntityVocabularyHelper entityVocabularyHelper;
	@Autowired private MessageSource messageSource;
	
	@Autowired private ImportResultRepository importResultRepo;
	
	@Autowired private ImportRepository importRepo;
	
	@Value("${baseUrl}")
	private String baseUrl;
	
	private PropertyChangeSupport support = new PropertyChangeSupport(this);
	
	private String importRunnerId;
	private Locale locale;
	
	boolean validateOnly;
	String template;
	String format;
	String body;
	String url; 
	String userId;
		
	String definition;
	Map<String, String> valueMap;
	
	ImportResult result;
	
	public void addPropertyChangeListener(PropertyChangeListener pcl) {
        support.addPropertyChangeListener(pcl);
    }
	
	public ImportRunner() {
		this.importRunnerId = UUID.randomUUID().toString();
	}
	
	@Override
	public void run() {
		try {
			this.doImport();
		} catch (Exception e) {
			log.error("Failed to run importer", e);
			support.firePropertyChange("error", null, e);
		}
	}
	
	public void doImport() throws IllegalArgumentException, MalformedURLException, IOException {
		this.result = new ImportResult();
		this.result.setImportRunnerId(this.getImportRunnerId());
		
		List<Entity> entities;
		if (url!=null && !url.isBlank()) {
			entities = entityService.readEntitiesFromURL(new URL(url), definition);
		} else if (format.equals("JSON")) {
			entities = entityService.readEntitiesFromJson(body, definition);
		} else {
			entities = entityService.readEntitiesFromYaml(body, definition);
		}
		
		if (entities==null || entities.isEmpty()) {
			support.firePropertyChange("result", null, this.result);
			importResultRepo.save(this.result);
			return;
		}
		
		
		// TODO: Debug why this is not working
		if (template!=null) {
			Optional<Entity> templateEntity = entityService.findLatestByEntityId(this.getTemplate(), true);
			if (templateEntity.isPresent()) {
				for (Entity e : entities) {
					e.getProperties().addAll(templateEntity.get().getProperties());
				}
			}
		}
		
		this.result.setEntities(entities);
		
		entityService.applyValueMappings(result.getEntities(), valueMap);
		
		EntityDefinition ed = entityDefService.findCurrentByName(definition, true);
		
		// Collect list of unknown properties per entity
		this.result.setUnknownPropertyLists(new ArrayList<>(result.getEntities().size()));
		
		// Collect list of resolved vocabulary entries per entity
		this.result.setResolvedVocabularyEntries(new ArrayList<>(result.getEntities().size()));
		for (Entity e : result.getEntities()) {
			this.result.getUnknownPropertyLists().add(propertyDefinitionHelper.mergeWithDefinition(e, ed, true));
			this.result.getResolvedVocabularyEntries().add(entityVocabularyHelper.resolveVocabularyEntries(e));
			
			// Merge vocabulary properties if configured
			entityService.autopopulateVocabularyData(e, ed);
		}
		
		// Collect list of validation violations per entity
		this.result.setValidationMessageLists(new ArrayList<>(result.getEntities().size()));
		try {
			List<ConstraintViolations> violations = entityService.validateEntities(ed, result.getEntities());
			for (ConstraintViolations v : violations) {
				List<ValidationViolationMessage> validationMessages = new ArrayList<>();
				if (!v.isValid()) {
					v.forEach(x -> validationMessages.add(new ValidationViolationMessage(
							this.convertValidationConstraintNameToViewPropertyLabel(x.name()), 
							x.messageKey(), 
							messageSource.getMessage("validation.violation." + x.messageKey(), x.args(), locale),
							x.args()[x.args().length-1])));
					
				}
				this.result.getValidationMessageLists().add(validationMessages);
			}
		} catch (ValidationConfigurationException ex) {
			log.error("Entity validation failed with ValidationConfigurationException", ex);
			//return this.buildFailedWithExceptionResponse(HttpStatus.INTERNAL_SERVER_ERROR, messageSource.getMessage("view.import.messages.validation_exception", null, locale), ex, req, resp, locale);
		}
		
		
		if (!validateOnly) {
			this.result.setIm(this.importEntities(this.result.getEntities(), this.result.getValidationMessageLists().stream().map(vl -> !vl.isEmpty()).toList(), ed, template));
		}
		importResultRepo.save(this.result);
		support.firePropertyChange("result", null, this.result);
	}
	
	
	public Import importEntities(List<Entity> entities, List<Boolean> isDraft, EntityDefinition ed, String templateId) {
		Import im = new Import();
		im.setTemplateId(templateId);
		im.setUserUniqueId(userId);
		im.setCreationInstant(Instant.now());
		
		importRepo.save(im);
		
		for (int i=0; i<entities.size(); i++) {

			entities.get(i).setUserUniqueId(userId);
			entities.get(i).setImportId(im.getUniqueId());
			entities.get(i).setReadOnly(true);
			entities.get(i).setDraft(isDraft.get(i));
			
			// Entities that are related by a vocabulary need to be persisted first 
			entityService.setOrCreateRelatedEntities(entities.get(i), ed);
			
			// Save entity
			entityService.save(entities.get(i));
		}
		
		return im;
	}
	
	private String convertValidationConstraintNameToViewPropertyLabel(String name) {
		String[] nameParts = name.split("\\.");
		StringBuilder propertyLabelBuilder = new StringBuilder();
		for (String part : nameParts) {
			if (!part.startsWith("_")) {			
				if (!propertyLabelBuilder.isEmpty()) {
					propertyLabelBuilder.append(".");
				}
				propertyLabelBuilder.append(part);
			}		
		}
		return propertyLabelBuilder.toString();
	}
}
