package de.uniba.minf.registry.controller.user;

import java.util.List;
import java.util.Locale;
import java.util.Optional;
import java.util.stream.Collectors;

import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.fasterxml.jackson.databind.node.TextNode;

import de.uniba.minf.auth.spring.mvc.model.AuthPojo;
import de.uniba.minf.core.rest.exception.ApiInsufficientPermissionsException;
import de.uniba.minf.core.rest.exception.ApiItemNotFoundException;
import de.uniba.minf.core.rest.helper.RestLinksHelper;
import de.uniba.minf.core.rest.model.RestItemResponse;
import de.uniba.minf.core.rest.model.RestItemsResponse;
import de.uniba.minf.core.rest.model.RestResponse;
import de.uniba.minf.core.rest.model.RestResponse.ApiActions;
import de.uniba.minf.registry.model.PersistedUser;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.servlet.http.HttpServletRequest;

@Tag(name = "Users", description = "API methods on users")
@RestController
@RequestMapping("/api/v1/users")
public class UserRestController extends BaseUserRestController<PersistedUser> {
	private static final String ITEM_TYPE = "user";
	
	public UserRestController() {
		super("/api/v1/users");
	}
	
	@GetMapping
	public RestResponse getUsers(HttpServletRequest request, Locale locale) throws ApiInsufficientPermissionsException, ApiItemNotFoundException {
		AuthPojo authPojo = authInfoHelper.getAuth();
		if(authPojo.getLevel()<100) {
			Optional<PersistedUser> authUser = userRepo.findById(authPojo.getUserId());
			if (authUser.isPresent()) {
				return this.getUserByUniqueId(authUser.get().getUniqueId(), request, locale);
			} else {
				throw new ApiItemNotFoundException(ITEM_TYPE, authPojo.getUserId());
			}
		}
		return this.getAllUsers(request, locale);
	}
	
	@GetMapping("/{userUniqueId}")
	public RestItemResponse getUserByUniqueId(@PathVariable("userUniqueId") String userUniqueId, HttpServletRequest request, Locale locale) throws ApiInsufficientPermissionsException, ApiItemNotFoundException {
		RestItemResponse itemResponse = new RestItemResponse();
		AuthPojo authPojo = authInfoHelper.getAuth();
		itemResponse.setItem(this.getUserItem(this.checkCanAccessUser(authPojo, userUniqueId), request.getRequestURL().toString(), false));
		itemResponse.setLinks(this.getLinks(request.getRequestURL().toString()));
		return itemResponse;
	}
	
	@PostMapping("/{userUniqueId}")
	public RestItemResponse saveUser(@PathVariable("userUniqueId") String userUniqueId, @RequestBody PersistedUser data, HttpServletRequest request, Locale locale) throws ApiInsufficientPermissionsException, ApiItemNotFoundException {
		AuthPojo authPojo = authInfoHelper.getAuth();
		if (!authPojo.getUserId().equals(userUniqueId)) {
			throw new ApiInsufficientPermissionsException();
		}
		
		Optional<PersistedUser> user = userRepo.findById(userUniqueId);
		RestItemResponse itemResponse = new RestItemResponse();
		if (user.isPresent()) {
			user.get().setEmail(data.getEmail());
			user.get().setFirstName(data.getFirstName());
			user.get().setLastName(data.getLastName());
		
			itemResponse.setAction(ApiActions.UPDATED);
			userRepo.save(user.get());
		}
		itemResponse.setItem(this.getUserItem(this.checkCanAccessUser(authPojo, userUniqueId), request.getRequestURL().toString(), false));
		itemResponse.setLinks(this.getLinks(request.getRequestURL().toString()));
		return itemResponse;
	}
	
	private RestItemsResponse getAllUsers(HttpServletRequest request, Locale locale) {		
		RestItemsResponse response = new RestItemsResponse();
		List<PersistedUser> users = userRepo.findAll().stream().map(u -> { /*u.setAccessTokens(null); */return u;} ).collect(Collectors.toList());
		
		response.setSize(users.size());
		response.setLinks(this.getLinks(request.getRequestURL().toString()));
		response.setItems(this.getUserItems(users, request.getRequestURL().toString()));
		return response;
	}
		
	private ArrayNode getUserItems(List<PersistedUser> users, String requestUrl) {
		ArrayNode array = objectMapper.createArrayNode();
		if (!users.isEmpty()) {
			for (PersistedUser user : users) {				
				array.add(this.getUserItem(user, requestUrl, true));
			}
		}
		return array;
	}
		
	private JsonNode getUserItem(PersistedUser user, String requestUrl, boolean suffixUniqueId) {
		JsonNode u = this.getItem(user, requestUrl, suffixUniqueId);
		
		if (u.has("accessTokens")) {
			ObjectNode objT;
			for (JsonNode t : (ArrayNode)u.path("accessTokens")) {
				objT = (ObjectNode)t;
				objT.set("_links", getTokenLink(requestUrl, user.getUniqueId(), objT.get("uniqueId").textValue()));
			}

		}
		return u;
	}
	
	private TextNode getTokenLink(String requestUrl, String userUniqueId, String tokenUniqueId) {
		StringBuilder tokenLinkBuilder = new StringBuilder();
		tokenLinkBuilder
			.append(requestUrl)
			.append(RestLinksHelper.SLASH)
			.append("tokens");
		
		if (tokenUniqueId!=null) {
			tokenLinkBuilder
				.append(RestLinksHelper.SLASH)
				.append(tokenUniqueId); 
		}
		return new TextNode(tokenLinkBuilder.toString());
	}

}