package de.uniba.minf.registry.controller;

import java.util.Collection;
import java.util.List;
import java.util.Locale;
import java.util.Optional;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.JsonMappingException;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.fasterxml.jackson.databind.node.TextNode;

import de.uniba.minf.core.rest.controller.BaseRestController;
import de.uniba.minf.core.rest.exception.ApiExecutionException;
import de.uniba.minf.core.rest.exception.ApiItemNotFoundException;
import de.uniba.minf.core.rest.model.RestItemResponse;
import de.uniba.minf.core.rest.model.RestItemsResponse;
import de.uniba.minf.core.rest.model.RestResponse.ApiActions;
import de.uniba.minf.registry.model.definition.EntityDefinition;
import de.uniba.minf.registry.model.entity.Entity;
import de.uniba.minf.registry.model.vocabulary.VocabularyDefinition;
import de.uniba.minf.registry.model.vocabulary.VocabularyEntry;
import de.uniba.minf.registry.model.vocabulary.VocabularyLookupException;
import de.uniba.minf.registry.model.vocabulary.VocabularyLookupService;
import de.uniba.minf.registry.repository.VocabularyDefinitionRepository;
import de.uniba.minf.registry.repository.VocabularyEntryRepository;
import de.uniba.minf.registry.view.helper.PropertyPreviewHelper;
import de.uniba.minf.registry.view.helper.PropertyPreviewHelper.PreviewProperty;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.servlet.http.HttpServletRequest;
import lombok.extern.slf4j.Slf4j;

@Slf4j
@Tag(name = "Vocabulary entries", description = "Records forming the content of defined vocabularies")
@RestController
@RequestMapping("/api/v1/v")
public class VocabularyEntryController extends BaseRestController<VocabularyEntry> {

	public VocabularyEntryController() {
		super("/api/v1/v");
	}

	@Autowired @Qualifier("yamlMapper") private ObjectMapper yamlMapper;
	@Autowired @Qualifier("jsonMapper") private ObjectMapper jsonMapper;
	
	@Autowired private VocabularyEntryRepository vocabularyRepo;
	@Autowired private VocabularyDefinitionRepository vocabularyDefinitionRepo;
	@Autowired private VocabularyLookupService vocabularyLookupService;
	
	@PostMapping(consumes = {"application/x-yaml", "text/yaml"})
	public void postYaml(@RequestBody String updatedApi, Locale locale) throws JsonMappingException, JsonProcessingException {
		
		VocabularyEntry e = yamlMapper.readValue(updatedApi, VocabularyEntry.class);
		
		vocabularyRepo.save(e);
		
		log.debug(e.getDefinitionName());
	}
	
	@PostMapping(consumes = "application/json")
	public void postJson(@RequestBody String updatedApi, Locale locale) throws JsonMappingException, JsonProcessingException {
		if (updatedApi.trim().startsWith("[")) {
			List<VocabularyEntry> vocabularyEntries = jsonMapper.readValue(updatedApi, new TypeReference<List<VocabularyEntry>>(){});
			vocabularyRepo.saveOrUpdateAll(vocabularyEntries);
			log.debug("saved: " + vocabularyEntries.size());
		} else {
			VocabularyEntry e = jsonMapper.readValue(updatedApi, VocabularyEntry.class);
			vocabularyRepo.saveOrUpdate(e);
			log.debug(e.getDefinitionName());
		}
	}
	
	@GetMapping
	public RestItemsResponse get(HttpServletRequest request, Locale locale) {
		RestItemsResponse response = new RestItemsResponse();
		
		List<VocabularyEntry> entries = vocabularyRepo.findAll();
		
		response.setSize(entries.size());
		response.setLinks(this.getLinks(request.getRequestURL().toString()));
		response.setItems(this.getItems(entries, request.getRequestURL().toString()));
		
		return response;
	}
	
	@GetMapping("{id}")
	public VocabularyEntry get(@PathVariable("id") String id, Locale locale) {
		return vocabularyRepo.findById(id).get();
	}
	
	@GetMapping("/{name}/")
	public RestItemsResponse getByDefinition(@PathVariable("name") String name, HttpServletRequest request, Locale locale) {
		RestItemsResponse response = new RestItemsResponse();
		
		List<VocabularyEntry> entries = vocabularyRepo.findByDefinition(name);
		
		response.setSize(entries.size());
		response.setLinks(this.getLinks(request.getRequestURL().toString()));
		response.setItems(this.getItems(entries, request.getRequestURL().toString()));
		
		return response;
	}
	
	@PostMapping("/{name}/fetch")
	public RestItemResponse fetchVocabularyEntry(@PathVariable("name") String name, String key, HttpServletRequest request, Locale locale) throws ApiItemNotFoundException {
		RestItemResponse response = new RestItemResponse();
		
		try {
			VocabularyEntry v = vocabularyLookupService.resolve(name, key);
			if (v==null) {
				throw new ApiItemNotFoundException("vocabularyEntry", name + ":" + key);
			}
			response.setLinks(this.getLinks(request.getRequestURL().toString()));
			response.setItem(this.getItem(v, request.getRequestURL().toString()));
		} catch (VocabularyLookupException e) {
			log.error("Error while manually fetching vocabulary entry", e);
		}
		return response;
	}
	
	@PostMapping("/{name}/clear")
	public RestItemsResponse clearVocabularyEntries(@PathVariable("name") String name, HttpServletRequest request, Locale locale) {
		vocabularyRepo.deleteByVocabularyName(name);
		
		RestItemsResponse response = new RestItemsResponse();
		response.setAction(ApiActions.DELETED);
		return response;
	}
	
	@PostMapping("/{name}/search")
	public RestItemsResponse searchVocabularyEntry(@PathVariable("name") String name, @RequestBody(required=false) JsonNode body, HttpServletRequest request, Locale locale) throws ApiExecutionException {
		RestItemsResponse response = new RestItemsResponse();
		if (body==null || !body.has("query") || body.get("query").asText().isEmpty()) {
			throw new ApiExecutionException("No query provided");
		}
		
		try {
			List<VocabularyEntry> v = vocabularyLookupService.search(name, body.get("query").asText());
			VocabularyDefinition vd = vocabularyDefinitionRepo.findCurrentByName(name);
			
			response.setLinks(this.getLinks(request.getRequestURL().toString()));
			response.setItems(this.getItems(v, request.getRequestURL().toString(), vd, body.has("ui") && body.get("ui").asBoolean()));
			//vocabularyLookupService.resolve(name, key);
		} catch (VocabularyLookupException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		return response;
	}
	
	@GetMapping("/{name}/get/{key}")
	public RestItemResponse getVocabularyEntry(@PathVariable("name") String name, @PathVariable String key, @RequestParam(required=false) boolean ui, HttpServletRequest request, Locale locale) throws ApiItemNotFoundException {
		RestItemResponse response = new RestItemResponse();
		Optional<VocabularyEntry> v = vocabularyRepo.findByDefinitionAndKey(name, key);
		if (v.isEmpty()) {
			throw new ApiItemNotFoundException("vocabularyEntry", key);
		}
		response.setLinks(this.getLinks(request.getRequestURL().toString()));
		response.setItem(this.getItem(v.get(), request.getRequestURL().toString()));
		//vocabularyLookupService.resolve(name, key);
		
		return response;
	}
	
	@PostMapping("/{name}/get")
	public RestItemResponse postVocabularyEntry(@PathVariable("name") String name, @RequestBody(required=false) JsonNode body, HttpServletRequest request, Locale locale) throws ApiItemNotFoundException, ApiExecutionException {
		RestItemResponse response = new RestItemResponse();
		if (body==null || !body.has("key") || body.get("key").asText().isEmpty()) {
			throw new ApiExecutionException("Key not provided");
		}
		Optional<VocabularyEntry> v = vocabularyRepo.findByDefinitionAndKey(name, body.get("key").asText());
		if (v.isEmpty()) {
			throw new ApiItemNotFoundException("vocabularyEntry", name + ":" + body.get("key").asText());
		}
		VocabularyDefinition vd = vocabularyDefinitionRepo.findCurrentByName(name);
		response.setLinks(this.getLinks(request.getRequestURL().toString()));
		response.setItem(this.getItem(v.get(), request.getRequestURL().toString(), true, vd, body.has("ui") && body.get("ui").asBoolean()));
		
		return response;
	}
	
	protected ArrayNode getItems(Collection<VocabularyEntry> items, String requestUrl, VocabularyDefinition vd, boolean addViewItems) {
		ArrayNode array = objectMapper.createArrayNode();
		if (items!=null) {
			for (VocabularyEntry rec : items) {
				array.add(this.getItem(rec, requestUrl, true, vd, addViewItems));
			}
		}
		return array;
	}

	protected ObjectNode getItem(VocabularyEntry item, String requestUrl, boolean suffixUniqueId, VocabularyDefinition vd, boolean addViewItems) {
		ObjectNode recNode = objectMapper.valueToTree(item);
		if (addViewItems) {
			List<PreviewProperty> previewProperties = PropertyPreviewHelper.get().extractSubproperties(vd.getPreviewProperties(), item.getProperties());
			recNode.set("_view", objectMapper.valueToTree(previewProperties));
			// TODO Change to localized value if available
			recNode.set("_display", new TextNode(item.getPrimaryValue() + " (" + item.getKey() + ")"));
		}
		recNode.set("_links", this.getItemLinks(item, requestUrl, suffixUniqueId));
		return recNode;
	}
}
