package de.uniba.minf.registry.controller;

import java.util.Collection;
import java.util.Locale;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonMappingException;
import com.fasterxml.jackson.databind.ObjectMapper;

import de.uniba.minf.core.rest.controller.BaseRestController;
import de.uniba.minf.core.rest.model.RestItemsResponse;
import de.uniba.minf.registry.model.definition.EntityDefinition;
import de.uniba.minf.registry.pojo.EntityDefinitionPojo;
import de.uniba.minf.registry.pojo.converter.EntityDefinitionConverter;
import de.uniba.minf.registry.repository.EntityDefinitionRepository;
import de.uniba.minf.registry.repository.EntityRepository;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.servlet.http.HttpServletRequest;
import lombok.extern.slf4j.Slf4j;

@Slf4j
@Tag(name = "Entity definitions", description = "Access to the datamodel specifications of registry entities")
@RestController
@RequestMapping("/api/v1/def/entities")
public class EntityDefinitionController extends BaseRestController<EntityDefinitionPojo> {

	public EntityDefinitionController() {
		super("/api/v1/def/entities");
	}

	@Autowired @Qualifier("yamlMapper") private ObjectMapper yamlMapper;
	@Autowired @Qualifier("jsonMapper") private ObjectMapper jsonMapper;
	
	@Autowired private EntityDefinitionRepository entityDefinitionRepo;
	@Autowired private EntityDefinitionConverter entityDefinitionConverter;
	
	@Autowired private EntityRepository entityRepo;
	
	
	@PostMapping(consumes = {"application/x-yaml", "text/yaml"})
	public void post(@RequestBody String updatedApi, Locale locale) throws JsonMappingException, JsonProcessingException {
		
		EntityDefinition ed = yamlMapper.readValue(updatedApi, EntityDefinition.class);
		
		entityDefinitionRepo.save(ed);
		
		log.debug(""+ ed.getVersion());
	}
	
	@GetMapping
	public RestItemsResponse get(HttpServletRequest request, Locale locale) {
		RestItemsResponse response = new RestItemsResponse();
		
		Collection<EntityDefinitionPojo> entityDefinitions = entityDefinitionConverter.convertPojos(entityDefinitionRepo.findAllLatest());
		for (EntityDefinitionPojo p : entityDefinitions) {
			p.setEntryCount(entityRepo.countByDefinition(p.getName(), authInfoHelper.getAuth().isAuth()));
		}
		
		response.setSize(entityDefinitions.size());
		response.setLinks(this.getLinks(request.getRequestURL().toString()));
		response.setItems(this.getItems(entityDefinitions, request.getRequestURL().toString()));
		
		return response;
	}
	
	@GetMapping("{id}")
	public EntityDefinitionPojo get(@PathVariable("id") String id, Locale locale) {
		return entityDefinitionConverter.convertPojo(entityDefinitionRepo.findById(id).get());
	}
	
	@GetMapping("/by-name/{name}")
	public Collection<EntityDefinitionPojo> getByName(@PathVariable("name") String name, Locale locale) {
		return entityDefinitionConverter.convertPojos(entityDefinitionRepo.findByName(name));
	}
	
	@DeleteMapping("/by-name/{name}")
	public void deleteByName(@PathVariable("name") String name, Locale locale) {
		entityDefinitionRepo.deleteByName(name);
	}
	
}
