package de.uniba.minf.registry.controller;

import java.io.IOException;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import jakarta.servlet.http.HttpServletResponse;

import org.javers.core.diff.Diff;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.http.HttpStatus;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonMappingException;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.BooleanNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import am.ik.yavi.core.ConstraintViolations;
import de.uniba.minf.core.rest.controller.BaseRestController;
import de.uniba.minf.core.rest.exception.ApiExecutionException;
import de.uniba.minf.core.rest.exception.ApiItemNotFoundException;
import de.uniba.minf.core.rest.model.ErrorRestResponse;
import de.uniba.minf.core.rest.model.RestActionResponse;
import de.uniba.minf.core.rest.model.RestActionResponse.ApiActionStatus;
import de.uniba.minf.core.rest.model.RestItemResponse;
import de.uniba.minf.core.rest.model.RestResponse.ApiActions;
import de.uniba.minf.core.rest.model.RestItemsResponse;
import de.uniba.minf.core.rest.model.RestResponse;
import de.uniba.minf.registry.importer.ImportRunner;
import de.uniba.minf.registry.importer.ImportService;
import de.uniba.minf.registry.model.ImportResult;
import de.uniba.minf.registry.model.definition.EntityDefinition;
import de.uniba.minf.registry.model.entity.Entity;
import de.uniba.minf.registry.model.validation.exception.ValidationConfigurationException;
import de.uniba.minf.registry.repository.EntityDefinitionRepository;
import de.uniba.minf.registry.repository.EntityRepository;
import de.uniba.minf.registry.service.EntityDefinitionService;
import de.uniba.minf.registry.service.EntityService;
import de.uniba.minf.registry.view.helper.PropertyDefinitionHelper;
import de.uniba.minf.registry.view.helper.PropertyPreviewHelper;
import de.uniba.minf.registry.view.helper.PropertyPreviewHelper.PreviewProperty;
import de.uniba.minf.registry.view.helper.PropertyViewItemCombiner;
import de.uniba.minf.registry.view.model.ValidationViolationMessage;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.servlet.http.HttpServletRequest;
import lombok.extern.slf4j.Slf4j;

@Slf4j
@Tag(name = "Entities", description = "Methods for reading and writing entities.")
@RestController
@RequestMapping("/api/v1/e")
public class EntityController extends BaseRestController<Entity> implements ApplicationContextAware
 {
		
	public EntityController() {
		super("/api/v1/e");
	}

	@Autowired private EntityRepository entityRepo;

	@Autowired private EntityDefinitionRepository entityDefRepo;
	@Autowired private EntityDefinitionService entityDefService;
	@Autowired private PropertyDefinitionHelper propertyDefinitionHelper;
	@Autowired private PropertyViewItemCombiner propertyViewItemCombiner;
	
	@Autowired private EntityService entityService;
	@Autowired private ImportService importService;

	private ApplicationContext appContext;
	
	@Override
	public void setApplicationContext(ApplicationContext appContext) throws BeansException {
		this.appContext = appContext;
	}
	
	@GetMapping
	public RestItemsResponse get(HttpServletRequest request, Locale locale) {
		RestItemsResponse response = new RestItemsResponse();
		
		List<Entity> entities = entityRepo.findAll();
		
		response.setSize(entities.size());
		response.setLinks(this.getLinks(request.getRequestURL().toString()));
		response.setItems(this.getItems(entities, request.getRequestURL().toString()));
		
		return response;
	}
	
	@GetMapping("/{id}")
	public Entity get(@PathVariable("id") String id, Locale locale) {
		Optional<Entity> entity = entityRepo.findLatestByEntityId(id);
		if (entity.isPresent()) {
			return entity.get();
		}
		return entityRepo.findById(id).orElse(null);
	}
	
	@GetMapping("/{name}/")
	public RestItemsResponse getByDefinition(@PathVariable("name") String name, @RequestParam(name="includeDrafts", defaultValue="false") boolean includeDrafts, HttpServletRequest request, Locale locale) {
		RestItemsResponse response = new RestItemsResponse();
		
		Collection<Entity> entities = entityRepo.findLatestByDefinition(name, authInfoHelper.getAuth().isAuth() || includeDrafts);
		
		response.setSize(entities.size());
		response.setLinks(this.getLinks(request.getRequestURL().toString()));
		response.setItems(this.getItems(entities, request.getRequestURL().toString()));
		
		return response;
	}
	
	@GetMapping("/i/{importId}/")
	public RestItemsResponse getByImportId(@PathVariable("importId") String importId, HttpServletRequest request, Locale locale) {
		RestItemsResponse response = new RestItemsResponse();
		
		Collection<Entity> entities = entityRepo.findByImportId(importId);
		
		response.setSize(entities.size());
		response.setLinks(this.getLinks(request.getRequestURL().toString()));
		response.setItems(this.getItems(entities, request.getRequestURL().toString()));
		
		return response;
	}
	
	@PostMapping("/{name}/search")
	public RestItemsResponse searchEntries(@PathVariable("name") String name, @RequestBody(required=false) JsonNode body, HttpServletRequest request, Locale locale) throws ApiExecutionException {
		RestItemsResponse response = new RestItemsResponse();
		if (body==null || !body.has("query") || body.get("query").asText().isEmpty()) {
			throw new ApiExecutionException("No query provided");
		}
		
		EntityDefinition ed = entityDefRepo.findCurrentByName(name);
		
		Collection<Entity> entities = new ArrayList<>();
		entityService.addAllNotContainingIdentifier(ed, entities, entityRepo.findLatestByDefinitionAndQuery(name, body.get("query").asText()));
		
		if (ed.hasAutoqueryProperties()) {
			entityService.addAllNotContainingIdentifier(ed, entities, entityService.createEntitiesByAutoquery(name, body.get("query").asText()));
			//entities.addAll(entityService.createEntitiesByAutoquery(name, body.get("query").asText()));
		}
		
		
		
		response.setSize(entities.size());
		response.setLinks(this.getLinks(request.getRequestURL().toString()));
		response.setItems(this.getItems(entities, request.getRequestURL().toString(), ed, body.has("ui") && body.get("ui").asBoolean()));
		
		return response;
	}
	
	@PostMapping("/{name}/get")
	public RestItemResponse getEntry(@PathVariable("name") String name, @RequestBody(required=false) JsonNode body, HttpServletRequest request, Locale locale) throws ApiItemNotFoundException, ApiExecutionException {
		RestItemResponse response = new RestItemResponse();
		String entityId = null;
		String uniqueId = null;
		
		if (body!=null && body.has("entityId") && !body.get("entityId").asText().isEmpty()) {
			entityId = body.get("entityId").asText();
		} else if (body!=null && body.has("uniqueId") && !body.get("uniqueId").asText().isEmpty()) {
			uniqueId = body.get("uniqueId").asText();
		} else {
			throw new ApiExecutionException("Neither EntityId or UniqueId provided");
		}

		Optional<Entity> entity;
		if (entityId!=null) {
			entity = entityRepo.findLatestByEntityId(entityId);
		} else {
			entity = entityRepo.findById(uniqueId);
		}
		
		if (entity.isEmpty()) {
			throw new ApiItemNotFoundException("entity", entityId==null ? uniqueId : entityId);
		}
		
		EntityDefinition ed = entityDefRepo.findCurrentByName(name);
		
		response.setLinks(this.getLinks(request.getRequestURL().toString()));
		response.setItem(this.getItem(entity.get(), request.getRequestURL().toString(), true, ed, body.has("ui") && body.get("ui").asBoolean()));
		
		return response;
	}
		
	@PostMapping(value={"/", "/{entityId}"}, consumes="application/json")
	public RestResponse postEntityForm(@PathVariable(required=false,value="entityId") String entityId, @RequestBody JsonNode formData, HttpServletRequest req, HttpServletResponse resp, Locale locale) throws ValidationConfigurationException {
		boolean validateOnly = req.getHeader("Entity-Validate-Only")!=null && req.getHeader("Entity-Validate-Only").equals("true");
		boolean changeToDraft = req.getHeader("Entity-Save-Draft")!=null && req.getHeader("Entity-Save-Draft").equals("true");
		boolean changeToPublished = req.getHeader("Entity-Save-Published")!=null && req.getHeader("Entity-Save-Published").equals("true");
		boolean copyAsTemplate = req.getHeader("Entity-Save-Template")!=null && req.getHeader("Entity-Save-Template").equals("true");

		Entity e = entityService.prepareEntity(entityId, this.getEntityDefinitionFromDataArray(formData), copyAsTemplate, changeToDraft, changeToPublished);		
		EntityDefinition ed = entityDefService.findCurrentByName(e.getDefinitionName(), true);
		
		// Merge entity properties with its definitions to be able to transform value types
		//  Since all form data is submitted as strings such transformations might be required
		propertyDefinitionHelper.mergeWithDefinition(e, ed, false);
		
		// Fill entity properties as submitted
		entityService.fillEntityFromDataArray(e, formData);
		
		// Merge vocabulary properties if configured
		entityService.autopopulateVocabularyData(e, ed);
		
		// Validate entity and return on violations of if validateOnly
		// Drafts are not validated unless validateOnly
		if (validateOnly || e.isPublished()) {
			try {
				ConstraintViolations violations = entityService.validateEntity(ed, e);
				// Return on violations
				if (!violations.isValid()) {
					return this.buildValidationFailedResponse(violations, req, resp, locale);
				} 
				// Return if validation only
				else if (validateOnly) {
					return this.buildItemResponse(ApiActions.VALIDATED, e);
				}
				// Set valid state if we continue
				e.setValid(violations.isValid());
			} catch (ValidationConfigurationException e1) {
				log.error("Entity validation failed with ValidationConfigurationException", e1);
				throw e1;
			}
		}		
		
		// Remerge with definition, discard unknown properties
		propertyDefinitionHelper.mergeWithDefinition(e, ed, true);
		
		// Entities that are related by a vocabulary need to be persisted first 
		entityService.setOrCreateRelatedEntities(e, ed);
				
		// Persist new entity
		if (e.getEntityId()==null && (e.getProperties()!=null && !e.getProperties().isEmpty())) {
			return buildItemResponse(ApiActions.CREATED, entityService.save(e));
		} else if (e.getEntityId()==null) {
			return buildItemResponse(ApiActions.UNCHANGED, e);
		}
				
		Diff diff = entityService.getChanges(e);
		if ((e.getProperties()!=null && !e.getProperties().isEmpty()) && 
				(diff.hasChanges() || changeToDraft || changeToPublished)) {
			return buildItemResponse(ApiActions.UPDATED, entityService.save(e));
		} else {
			return buildItemResponse(ApiActions.UNCHANGED, e);
		}
	}
	
	@DeleteMapping("/{entityId}")
	public RestResponse deleteEntity(@PathVariable(required=false,value="entityId") String entityId, HttpServletResponse resp, Locale locale) {
		Optional<Entity> eExist = entityRepo.findLatestByEntityId(entityId);
		if (eExist.isPresent()) {
			eExist.get().setDeleted(true);
			entityRepo.save(eExist.get());
		}
		log.debug(entityId);
		
		RestItemResponse response = new RestItemResponse();
		response.setAction(ApiActions.DELETED);
		return response;
	}
	
	@PostMapping(value="/{name}/validate", consumes={"application/json"})
	public RestResponse validateEntities(@PathVariable("name") String name, @RequestBody String body, HttpServletRequest req, HttpServletResponse resp, Locale locale) throws JsonProcessingException, ValidationConfigurationException {
		return this.validateOrImportEntities(name, body, req, resp, locale, true);
	}
	
	
	@PostMapping(value="/{name}/import", consumes={"application/json"})
	public RestResponse importEntities(@PathVariable("name") String name, @RequestBody String body, HttpServletRequest req, HttpServletResponse resp, Locale locale) throws ValidationConfigurationException, JsonMappingException, JsonProcessingException {
		return this.validateOrImportEntities(name, body, req, resp, locale, false);
	}
	
	@GetMapping(value={"/{name}/validate/{importId}","/{name}/import/{importId}"})
	public RestResponse validateEntities(@PathVariable("name") String name, @PathVariable("importId") String importId, HttpServletRequest req) {
		
		ApiActionStatus status = importService.getStatus(importId);
		
		if (status==ApiActionStatus.BUSY) {
			return new RestActionResponse(ApiActionStatus.BUSY, req.getRequestURI());
		} 
		
		ImportResult result = importService.getImportResult(importId);
		RestItemsResponse response = new RestItemsResponse();
		if (result==null || result.getIm()==null) {
			response.setAction(ApiActions.UNCHANGED);
		} else {
			response.setAction(ApiActions.CREATED);
		}
		

		// Remove definitions as they are not required for the response
		propertyDefinitionHelper.removeDefinitionsFromEntities(result.getEntities());
		
		response.setItems(this.getItems(result.getEntities(), req.getRequestURI()));
						
		ObjectNode dataNode = objectMapper.createObjectNode();
		dataNode.set("success", BooleanNode.valueOf(result.getValidationMessageLists().stream().anyMatch(vl -> !vl.isEmpty())));
		dataNode.set("validationErrors", objectMapper.valueToTree(result.getValidationMessageLists()));
		dataNode.set("unknownProperties", objectMapper.valueToTree(result.getUnknownPropertyLists()));
		dataNode.set("resolvedVocabularyEntries", objectMapper.valueToTree(result.getResolvedVocabularyEntries()));
		
		response.setServerData(dataNode);
		
		return response;		
	}
	
	private RestResponse validateOrImportEntities(String name, String envelope, HttpServletRequest req, HttpServletResponse resp, Locale locale, boolean validateOnly) throws JsonMappingException, JsonProcessingException {
		JsonNode n = objectMapper.readTree(envelope);
		
		String template = (n.has("template") && !n.get("template").asText().isBlank()) ? n.get("template").asText() : null;
		String format = (n.has("format") && n.get("format").asText().equals("yaml")) ? "YAML" : "JSON";
		String body = n.has("data") ? n.get("data").asText() : null;
		String url = n.has("importUrl") ? n.get("importUrl").asText() : null;
				
		Map<String, String> valueMap = new HashMap<>();
		
		if (n.has("mappings")) {
			for (JsonNode nM : n.get("mappings")) {
				if (!nM.isArray() || nM.size()<2) {
					continue;
				}
				valueMap.put(nM.get(0).asText(), nM.get(1).asText());
			}
		}
		
		if (url!=null && !url.isBlank()) {
			try {
				URL u = new URL(url);
				HttpURLConnection huc =  ( HttpURLConnection )  u.openConnection (); 
				huc.setRequestMethod ("HEAD");
				huc.setInstanceFollowRedirects(true);
				huc.connect(); 
				int code = huc.getResponseCode() ;
				
				if (code < 200 || code > 400) {
					return this.buildFailedWithExceptionResponse(HttpStatus.valueOf(code), messageSource.getMessage("view.import.messages.url_invalid", null, locale), null, req, resp, locale);
				}
			} catch (MalformedURLException e) {
				return this.buildFailedWithExceptionResponse("import-data-url", "import-data-url", messageSource.getMessage("view.import.messages.url_invalid", null, locale), e, req, resp, locale);
			} catch (IOException e) {
				return this.buildFailedWithExceptionResponse(HttpStatus.BAD_REQUEST, messageSource.getMessage("view.import.messages.url_invalid", null, locale) + ": " + e.getLocalizedMessage(), null, req, resp, locale);
			}
			
		}
		
		ImportRunner ir = appContext.getBean(ImportRunner.class);
		ir.setUrl(url);
		ir.setFormat(format);
		ir.setBody(body);
		ir.setValidateOnly(validateOnly);
		ir.setTemplate(template);
		ir.setValueMap(valueMap);
		ir.setDefinition(name);
		ir.setUserId(authInfoHelper.getUserId());
		
		importService.importAsync(ir);
		
		return new RestActionResponse(ApiActionStatus.STARTED, req.getRequestURI() + "/" + ir.getImportRunnerId());
	}
	
	
	/*public void buildTheResponse() {
		
		if (!validateOnly) {
			RestItemResponse response = new RestItemResponse();
			response.setAction(ApiActions.CREATED);
			
			Import im = importService.importEntities(entities, validationMessageLists.stream().map(vl -> !vl.isEmpty()).toList(), ed, readonly, template);
			ObjectNode recNode = objectMapper.valueToTree(im);
			
			ObjectNode links = objectMapper.createObjectNode();
			links.set("self", new TextNode(this.getBaseUrl() + "/api/v1/imports" + RestLinksHelper.SLASH + im.getUniqueId()));
			
			recNode.set("_links", links);
			
			response.setItem(recNode);
			
			
			this.response = response;
			return;
		}
		
		RestItemsResponse response = new RestItemsResponse();
		response.setAction(ApiActions.UNCHANGED);

		// Remove definitions as they are not required for the response
		propertyDefinitionHelper.removeDefinitionsFromEntities(entities);
		
		//response.setItems(this.getItems(entities, requestUrl));
						
		ObjectNode dataNode = objectMapper.createObjectNode();
		dataNode.set("success", BooleanNode.valueOf(validationMessageLists.stream().anyMatch(vl -> !vl.isEmpty())));
		dataNode.set("validationErrors", objectMapper.valueToTree(validationMessageLists));
		dataNode.set("unknownProperties", objectMapper.valueToTree(unknownPropertyLists));
		dataNode.set("resolvedVocabularyEntries", objectMapper.valueToTree(resolvedVocabularyEntries));
		
		response.setServerData(dataNode);
		
		this.response = response;
	}
	*/
	
	private String getEntityDefinitionFromDataArray(JsonNode formData) {
		for (JsonNode fieldNode : formData) {
			if (fieldNode.get("name").asText().equals("_entity")) {
				return fieldNode.get("value").asText();
			}
		}
		return null;
	}
		
	private RestItemResponse buildItemResponse(ApiActions action, Entity e) {
		RestItemResponse response = new RestItemResponse();
		response.setAction(action);
		response.setItem(objectMapper.valueToTree(e));
		return response;
	}
		
	private ErrorRestResponse buildValidationFailedResponse(ConstraintViolations violations, HttpServletRequest req, HttpServletResponse resp, Locale locale) {
		ErrorRestResponse response = new ErrorRestResponse(HttpStatus.BAD_REQUEST.value(), messageSource.getMessage("view.import.messages.validation_result_invalid", null, locale), req.getRequestURL().toString());
		List<ValidationViolationMessage> validationMessages = new ArrayList<>();
		violations.forEach(x -> validationMessages.add(new ValidationViolationMessage(
				this.convertValidationConstraintNameToViewPropertyLabel(x.name()), 
				x.messageKey(), 
				messageSource.getMessage("validation.violation." + x.messageKey(), x.args(), locale),
				x.args()[x.args().length-1])));
		response.setErrorDetails(objectMapper.valueToTree(validationMessages));
		
		resp.setStatus(HttpStatus.BAD_REQUEST.value());
		return response;
	}
	
	private ErrorRestResponse buildFailedWithExceptionResponse(HttpStatus code, String message, Exception ex, HttpServletRequest req, HttpServletResponse resp, Locale locale) {
		return this.buildFailedWithExceptionResponse(code, null, null, message, ex, req, resp, locale);
	}
	
	private ErrorRestResponse buildFailedWithExceptionResponse(String name, String key, String message, Exception ex, HttpServletRequest req, HttpServletResponse resp, Locale locale) {
		return this.buildFailedWithExceptionResponse(HttpStatus.BAD_REQUEST, name, key, message, ex, req, resp, locale);
	}
	
	private ErrorRestResponse buildFailedWithExceptionResponse(HttpStatus code, String name, String key, String message, Exception ex, HttpServletRequest req, HttpServletResponse resp, Locale locale) {
		ErrorRestResponse response = new ErrorRestResponse(code.value(), message, req.getRequestURL().toString());
		if (name!=null || key!=null) {
			List<ValidationViolationMessage> validationMessages = new ArrayList<>();
			validationMessages.add(new ValidationViolationMessage(name, key, ex==null ? messageSource.getMessage("view.import.messages.generic_exception", null, locale) : ex.getLocalizedMessage(), null));
			response.setErrorDetails(objectMapper.valueToTree(validationMessages));
		}
		resp.setStatus(code.value());
		return response;
	}
	
	private String convertValidationConstraintNameToViewPropertyLabel(String name) {
		String[] nameParts = name.split("\\.");
		StringBuilder propertyLabelBuilder = new StringBuilder();
		for (String part : nameParts) {
			if (!part.startsWith("_")) {			
				if (!propertyLabelBuilder.isEmpty()) {
					propertyLabelBuilder.append(".");
				}
				propertyLabelBuilder.append(part);
			}		
		}
		return propertyLabelBuilder.toString();
	}
	
	private ArrayNode getItems(Collection<Entity> items, String requestUrl, EntityDefinition ed, boolean addViewItems) {
		ArrayNode array = objectMapper.createArrayNode();
		if (items!=null) {
			for (Entity rec : items) {
				array.add(this.getItem(rec, requestUrl, true, ed, addViewItems));
			}
		}
		return array;
	}

	private ObjectNode getItem(Entity item, String requestUrl, boolean suffixUniqueId, EntityDefinition ed, boolean addViewItems) {
		ObjectNode recNode = objectMapper.valueToTree(item);
		if (addViewItems) {
			List<PreviewProperty> previewProperties = PropertyPreviewHelper.get().extractSubproperties(ed.getPreviewProperties(), item.getProperties());
			recNode.set("_view", objectMapper.valueToTree(previewProperties));
			recNode.set("_display", objectMapper.valueToTree(PropertyPreviewHelper.get().getDisplayString(previewProperties, item.getEntityId())));
		}
		recNode.set("_links", this.getItemLinks(item, requestUrl, suffixUniqueId));
		return recNode;
	}
}
