package de.uniba.minf.registry.config.db;

import static org.springframework.data.mongodb.core.query.Criteria.where;
import static org.springframework.data.mongodb.core.query.Query.query;
import static org.springframework.data.mongodb.core.query.Update.update;

import java.time.Instant;
import java.util.UUID;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Configuration;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.mapping.event.AbstractMongoEventListener;
import org.springframework.data.mongodb.core.mapping.event.AfterSaveEvent;
import org.springframework.data.mongodb.core.mapping.event.BeforeConvertEvent;
import org.springframework.web.context.request.RequestAttributes;
import org.springframework.web.context.request.RequestContextHolder;

import de.uniba.minf.auth.spring.mvc.AuthInfoHelper;
import de.uniba.minf.registry.model.entity.Entity;

@Configuration
public class EntityEventListener extends AbstractMongoEventListener<Entity> {
	
	@Autowired private AuthInfoHelper authinfoHelper;
	@Autowired private MongoTemplate mongoTemplate;
		
	@Override
	public void onBeforeConvert(BeforeConvertEvent<Entity> event) {
		super.onBeforeConvert(event);
		
		RequestAttributes attrs = RequestContextHolder.getRequestAttributes();
		String userUniqueId;
		if (attrs != null) {
			userUniqueId = authinfoHelper.getAuth().getUserId();
		} else {
			userUniqueId = "_system";
		}
		
		Entity obj = event.getSource();
		if (obj.getEntityId()==null) {
			obj.setEntityId(UUID.randomUUID().toString());
		}
		obj.setCreationInstant(Instant.now());
		obj.setUserUniqueId(userUniqueId);
	}
		
	@Override
	public void onAfterSave(AfterSaveEvent<Entity> event) {
		super.onAfterSave(event);
		
		// Update all previous versions of the entity that have no nextVersionUniqueId 
		//  with a nextVersionUniqueId that points to the now latest version
		// Using updateMulti to prevent loops between onBeforeConvert and onAfterSave
		mongoTemplate.updateMulti(
				query(
						where("entityId").is(event.getSource().getEntityId())
							.and("nextVersionUniqueId").isNull()
							.and("uniqueId").ne(event.getSource().getUniqueId())),
				update("nextVersionUniqueId", event.getSource().getUniqueId()),
				Entity.class);
	}
}