package de.uniba.minf.registry.config.db;

import static org.springframework.data.mongodb.core.query.Criteria.where;
import static org.springframework.data.mongodb.core.query.Query.query;
import static org.springframework.data.mongodb.core.query.Update.update;

import java.time.Instant;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Configuration;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.mapping.event.AbstractMongoEventListener;
import org.springframework.data.mongodb.core.mapping.event.AfterSaveEvent;
import org.springframework.data.mongodb.core.mapping.event.BeforeConvertEvent;
import org.springframework.web.context.request.RequestAttributes;
import org.springframework.web.context.request.RequestContextHolder;

import de.uniba.minf.auth.spring.mvc.AuthInfoHelper;
import de.uniba.minf.registry.model.definition.BaseDefinition;
import de.uniba.minf.registry.model.definition.EntityDefinition;
import de.uniba.minf.registry.model.vocabulary.VocabularyDefinition;
import de.uniba.minf.registry.repository.EntityDefinitionRepository;
import de.uniba.minf.registry.repository.VocabularyDefinitionRepository;

@Configuration
public class BaseDefinitionEventListener extends AbstractMongoEventListener<BaseDefinition> {
	
	@Autowired private AuthInfoHelper authinfoHelper;

	@Autowired private VocabularyDefinitionRepository vocabularyDefinitionRepository;
	@Autowired private EntityDefinitionRepository entityDefinitionRepository;

	@Autowired private MongoTemplate mongoTemplate;
	
	@Override
	public void onBeforeConvert(BeforeConvertEvent<BaseDefinition> event) {
		super.onBeforeConvert(event);
		RequestAttributes attrs = RequestContextHolder.getRequestAttributes();
		String userUniqueId;
		if (attrs != null) {
			userUniqueId = authinfoHelper.getAuth().getUserId();
		} else {
			userUniqueId = "_system";
		}

		BaseDefinition def = event.getSource();
		def.setCreationInstant(Instant.now());
		def.setUserUniqueId(userUniqueId);
		
		BaseDefinition cDef = null;
		if (VocabularyDefinition.class.isAssignableFrom(event.getSource().getClass())) {
			cDef = vocabularyDefinitionRepository.findCurrentByName(event.getSource().getName());
		} else if (EntityDefinition.class.isAssignableFrom(event.getSource().getClass())) {
			cDef = entityDefinitionRepository.findCurrentByName(event.getSource().getName());
		}
		if (cDef!=null) {
			def.setVersion(cDef.getVersion()+1);
		} else {
			def.setVersion(1);
		}
	}
	
	@Override
	public void onAfterSave(AfterSaveEvent<BaseDefinition> event) {
		super.onAfterSave(event);
		
		// Update all previous versions of the definition that have no nextVersionUniqueId 
		//  with a nextVersionUniqueId that points to the now latest version
		// Using updateMulti to prevent loops between onBeforeConvert and onAfterSave
		mongoTemplate.updateMulti(
				query(
						where("name").is(event.getSource().getName())
							.and("nextVersionUniqueId").isNull()
							.and("uniqueId").ne(event.getSource().getUniqueId())),
				update("nextVersionUniqueId", event.getSource().getUniqueId()),
				VocabularyDefinition.class.isAssignableFrom(event.getSource().getClass()) ? VocabularyDefinition.class : EntityDefinition.class);
	}
}