package de.uniba.minf.registry.config;

import java.util.Arrays;
import java.util.Locale;

import org.springframework.beans.BeansException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.boot.web.server.WebServerFactoryCustomizer;
import org.springframework.boot.web.servlet.server.ConfigurableServletWebServerFactory;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.io.Resource;
import org.springframework.core.io.ResourceLoader;
import org.springframework.format.FormatterRegistry;
import org.springframework.format.datetime.DateFormatter;
import org.springframework.web.servlet.LocaleResolver;
import org.springframework.web.servlet.config.annotation.InterceptorRegistry;
import org.springframework.web.servlet.config.annotation.ResourceHandlerRegistry;
import org.springframework.web.servlet.config.annotation.WebMvcConfigurer;
import org.springframework.web.servlet.i18n.SessionLocaleResolver;
import org.thymeleaf.spring6.SpringTemplateEngine;
import org.thymeleaf.spring6.templateresolver.SpringResourceTemplateResolver;
import org.thymeleaf.spring6.view.ThymeleafViewResolver;
import org.thymeleaf.templatemode.TemplateMode;

import de.uniba.minf.auth.ProfileActionHandler;
import de.uniba.minf.auth.spring.controller.LoginController;
import de.uniba.minf.auth.spring.controller.LogoutController;
import de.uniba.minf.auth.spring.controller.SAMLMetadataController;
import de.uniba.minf.registry.profiles.UserProfileActionHandler;
import de.unibamberg.minf.core.web.config.LocalizationConfig;
import de.unibamberg.minf.core.web.init.LocaleAwareInitializationService;
import de.unibamberg.minf.core.web.init.LocaleAwareInitializationServiceImpl;
import de.unibamberg.minf.core.web.interceptor.UserLocaleChangeInterceptor;
import de.unibamberg.minf.core.web.localization.MessageSource;
import de.unibamberg.minf.core.web.theming.ThemeManagerImpl;
import jakarta.annotation.PostConstruct;
import lombok.Getter;
import lombok.Setter;
import lombok.extern.slf4j.Slf4j;
import nz.net.ultraq.thymeleaf.layoutdialect.LayoutDialect;
import nz.net.ultraq.thymeleaf.layoutdialect.decorators.strategies.GroupingStrategy;

@Slf4j
@Configuration
@ConfigurationProperties
public class WebConfig implements WebMvcConfigurer, ApplicationContextAware {
	@Getter @Setter private String contextPath = "";
	@Getter @Setter private String theme = "default-theme";
	@Getter @Setter private LocalizationConfig localization;

	@Getter private String themeDirectory;
	
	@Autowired ResourceLoader resourceLoader;
	
	private ApplicationContext applicationContext;

	@Override
	public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
		this.applicationContext = applicationContext;
	}
	
	@PostConstruct
	public void completeConfiguration() {
		if (theme.startsWith("file:")) {
			themeDirectory = theme + (theme.endsWith("/") ? "" : "/");
		} else {
			themeDirectory = "classpath:themes/" + theme + "/";
		}
		Resource themeDirectoryResource = resourceLoader.getResource(themeDirectory);
		if (!themeDirectoryResource.exists()) {
			log.error("Invalid theme specified: cannot find '{}'", themeDirectory);
			log.warn("Falling back to default-theme");
			theme = "default-theme";
			themeDirectory = "classpath:themes/" + theme + "/";
		} else {
			log.info("Theme directory set to: {}", themeDirectory);
		}
		
		if (localization.getBaseNames()==null) {
			localization.setBaseNames("classpath:i18n/messages", themeDirectory + "i18n/theme");
		}
	}
	
	/**
	 * WebServerFactoryCustomizer bean that adapts to a configured context path for the application. This adaption is not 
	 *  necessary for implementation of the dariahsp-core library, but helps with setting up the application as it might 
	 *  be available or proxied at their deployments
	 * 
	 * @return WebServerFactoryCustomizer
	 */
	@Bean
	public WebServerFactoryCustomizer<ConfigurableServletWebServerFactory> webServerFactoryCustomizer() {
		log.info("Web server context path set to {}", contextPath.isEmpty() ? "/" : contextPath);
	    return factory -> factory.setContextPath(contextPath);
	}
	
	@Bean
	public ThemeManagerImpl themeManagerImpl() {
		ThemeManagerImpl themeManagerImpl = new ThemeManagerImpl();
		themeManagerImpl.setTheme(themeDirectory);
		//themeManagerImpl.setCheckExistsSubpaths(Arrays.asList("/i18n/theme_de.properties", "/css/bootstrap.css", "/img/theme-logo-de.svg", "/js", "/jsp/footer.jsp"));
		themeManagerImpl.setCheckExistsSubpaths(Arrays.asList("/i18n/theme_de.properties", "/img/theme-logo.svg"));
		return themeManagerImpl;
	} 
	
	@Override
	public void addInterceptors(InterceptorRegistry registry) {
	    registry.addInterceptor(localeChangeInterceptor());
	}
	
	@Bean
    public SpringResourceTemplateResolver templateResolver(){
        SpringResourceTemplateResolver templateResolver = new SpringResourceTemplateResolver();
        templateResolver.setApplicationContext(this.applicationContext);
        templateResolver.setPrefix("/WEB-INF/views/");
        templateResolver.setSuffix(".html");
        templateResolver.setTemplateMode(TemplateMode.HTML);
        templateResolver.setCacheable(false);
        templateResolver.setCharacterEncoding("UTF-8");
        return templateResolver;
    }

    @Bean
    public SpringTemplateEngine templateEngine(){
        SpringTemplateEngine templateEngine = new SpringTemplateEngine();
        templateEngine.setTemplateResolver(templateResolver());
        templateEngine.setEnableSpringELCompiler(true);
        templateEngine.setTemplateEngineMessageSource(messageSource());
        templateEngine.addDialect(new LayoutDialect(new GroupingStrategy()));
        return templateEngine;
    }

    @Bean
    public ThymeleafViewResolver viewResolver(){
        ThymeleafViewResolver viewResolver = new ThymeleafViewResolver();
        viewResolver.setTemplateEngine(templateEngine());
        viewResolver.setCharacterEncoding("UTF-8");
        viewResolver.setOrder(1);
        return viewResolver;
    }
    
	@Bean
	public MessageSource messageSource() {
		MessageSource messageSource = new MessageSource();
		messageSource.setBasenames(localization.getBaseNames());
		messageSource.setLoggingMode(localization.isDebug());
		messageSource.setCacheSeconds(localization.getCacheSeconds());
		messageSource.setHideMissingDynamic(localization.isHideMissingDynamic());
		messageSource.setReturnMissingCodes(true);
		messageSource.setDefaultEncoding("UTF-8");
		
		return messageSource;
	}
	
	@Bean
	public LocaleAwareInitializationService initService() {
		LocaleAwareInitializationServiceImpl initService = new LocaleAwareInitializationServiceImpl();
		initService.setBasename(localization.getBaseNames()[0]);
		
		log.debug("Initialized LocaleAwareInitializationServiceImpl");
		
	    return initService;
	}
	
	@Bean
	public LocaleResolver localeResolver() {
	    final SessionLocaleResolver localeResolver = new SessionLocaleResolver();
	    localeResolver.setDefaultLocale(new Locale("de", "DE"));
	    return localeResolver;
	}

	
	@Bean
	public UserLocaleChangeInterceptor localeChangeInterceptor() {
		UserLocaleChangeInterceptor lci = new UserLocaleChangeInterceptor();
		lci.setInitService(initService());
	    lci.setParamName("lang");
	    return lci;
	}
    
    @Override
    public void addResourceHandlers(ResourceHandlerRegistry registry) {
    	registry.addResourceHandler("/webjars/**").addResourceLocations("/webjars/");
        registry.addResourceHandler("/resources/**").addResourceLocations("/resources/");
        
        if (theme.startsWith("file:")) {
        	registry.addResourceHandler("/theme/**").addResourceLocations(theme + "/");
		} else {
			registry.addResourceHandler("/theme/**").addResourceLocations("classpath:themes/" + theme + "/");
		}
              
        registry.addResourceHandler("/favicon.ico").addResourceLocations("/resources/favicon.ico");
    }
    
    @Override
    public void addFormatters(final FormatterRegistry registry) {
        //super.addFormatters(registry);
        registry.addFormatter(dateFormatter());
    }

    @Bean
    public DateFormatter dateFormatter() {
        return new DateFormatter();
    }
    
	/**
	 * Bean that is injected into {@link CustomizableProfileManager} to facilitate observation of login and logout actions
	 *  Implementations can provide custom implementations of the {@link ProfileActionHandler} interface e.g. to log such
	 *  actions into a database
	 *    
	 * @return SampleProfileActionHandler bean
	 */
	@Bean
	public ProfileActionHandler profileActionPostprocessor() {
		return new UserProfileActionHandler();
	}
    
	/**
	 * Controller bean that facilitates access to generated or stored SAML SP metadata
	 * 
	 * @return SAMLMetadataController bean
	 */
	@Bean
	public SAMLMetadataController samlMetadataController() {
		return new SAMLMetadataController();
	}
	
	/**
	 * Controller that binds to common login mappings 
	 * 
	 * @return LoginLogoutController bean
	 */
	@Bean
	public LoginController loginController() {
		return new LoginController();
	}
	
	@Bean
	public LogoutController logoutController() {
		return new LogoutController();
	}
}
