package de.uniba.minf.registry.view.helper;

import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import de.uniba.minf.registry.model.Property;
import de.uniba.minf.registry.model.PropertyValue;
import de.uniba.minf.registry.model.TextPropertyValue;
import de.uniba.minf.registry.model.definition.EntityDefinition;
import de.uniba.minf.registry.model.definition.PropertyDefinition;
import de.uniba.minf.registry.model.definition.VocabularyPropertyDefinition;
import de.uniba.minf.registry.model.entity.Entity;
import de.uniba.minf.registry.model.entity.AutoqueryEntityLookupService;
import de.uniba.minf.registry.model.vocabulary.VocabularyEntry;
import de.uniba.minf.registry.model.vocabulary.VocabularyLookupException;
import de.uniba.minf.registry.model.vocabulary.VocabularyLookupService;
import de.uniba.minf.registry.repository.EntityDefinitionRepository;
import lombok.extern.slf4j.Slf4j;

@Slf4j
@Component
public class EntityVocabularyHelper {	
	@Autowired private VocabularyLookupService vocabularyLookupService;
	@Autowired private AutoqueryEntityLookupService entityLookupService;
	
	@Autowired private PropertyDefinitionHelper propertyDefinitionHelper;
	@Autowired private EntityDefinitionRepository entityDefRepo;
	
	@Value("#{${debug}==true || ${debugging.resolution}==true}")
	private boolean debug;
	
	private Map<ResolveCacheKey, List<VocabularyEntry>> resolveCache = null;
	private Instant cacheCreation = null;
	
	
	public List<ResolvedVocabularyEntry> resolveVocabularyEntries(Entity e) {
		if (resolveCache==null || ChronoUnit.HOURS.between(cacheCreation, Instant.now())>24) {
			resolveCache = new HashMap<>();
			cacheCreation = Instant.now();
		}		
		
		if (debug) {
			StringBuilder strBldr = new StringBuilder();
			resolveCache.entrySet().stream().forEach(en -> { 
				strBldr.append(en.getKey().getVocabulary());
				strBldr.append(" | ");
				strBldr.append(en.getKey().getQuery());
				strBldr.append(" => ");
				strBldr.append(en.getValue().size());
				strBldr.append(" candidate(s)\n");
			}); 
			log.debug("Current cache status\n" + strBldr.toString());
		}
		
		List<ResolvedVocabularyEntry> resolved = new ArrayList<>();
		if (e.getProperties()==null) {
			return resolved;
		}
		if (e.getPropertyDefinitions()==null) {
			EntityDefinition ed = entityDefRepo.findCurrentByName(e.getDefinitionName());
			propertyDefinitionHelper.mergeWithDefinition(e, ed, false);
		}
		this.resolveVocabularyEntries(e.getProperties(), e.getPropertyDefinitions(), resolved, null);
		return resolved;
	}
	
	private void resolveVocabularyEntries(List<Property> properties, List<PropertyDefinition> propertyDefinitions, List<ResolvedVocabularyEntry> resolved, String path) {
		for (PropertyDefinition pd : propertyDefinitions) {
			if (pd.isSimple()) {
				continue;
			}
			List<Property> subProperties = properties.stream().filter(p -> p.getLabel().equals(pd.getName())).toList();
			String subPath = path==null ? pd.getName() : path + "." + pd.getName();
			if (pd.isVocabulary()) {
				this.resolveVocabularyEntries(subProperties, VocabularyPropertyDefinition.class.cast(pd), resolved, path);
			} else if (pd.isHierarchical()) {
				for (Property p : subProperties) {
					if (p.getProperties()!=null) {
						for (int i=0; i<p.getProperties().size(); i++) {
							this.resolveVocabularyEntries(p.getProperties().get(i).getProperties(), p.getProperties().get(i).getPropertyDefinitions(), resolved, subPath + "[" + i + "]");
						}
					}
				}
			}
		}
	}

	private void resolveVocabularyEntries(List<Property> properties, VocabularyPropertyDefinition definition, List<ResolvedVocabularyEntry> resolved, String path) {
		List<PropertyValue> values;
				
		path = path==null ? definition.getName() : "." + definition.getName();
		
		PropertyValue v;
		String subpath;
		String query;
		for (Property p : properties) {
			values = p.valuesAsList();
			subpath = path;
			
			for (int i=0; i<values.size(); i++) {
				v = values.get(i);
				query = v.asText();
				
				if (values.size()>1) {
					subpath = path + "[" + i + "]";
				}
				
				try {
					boolean fromCache = false;
					List<VocabularyEntry> veCandidates = resolveCache.get(ResolveCacheKey.get(definition.getVocabulary(), query));
					if (veCandidates!=null) {
						log.debug("From cache '{}' in vocabulary '{}'", query, definition.getVocabulary());
						fromCache = true;
					} else {
						VocabularyEntry ve = null;
						if (definition.isEntity()) {
							ve = entityLookupService.resolve(definition.getVocabulary(), query);
						} else {
							ve = vocabularyLookupService.resolve(definition.getVocabulary(), query);
						}
						
						if (ve!=null) {
							veCandidates = new ArrayList<>();
							veCandidates.add(ve);
						} else if (definition.isEntity()) {
							veCandidates = entityLookupService.search(definition.getVocabulary(), query);
						} else { 
							veCandidates = vocabularyLookupService.search(definition.getVocabulary(), query);
						}
					}
					
					// Vocabulary query found no matching candidate
					if (veCandidates==null || veCandidates.isEmpty()) {
						log.debug("Resolving '{}' against vocabulary '{}': No match found", query, definition.getVocabulary());
						resolved.add(new ResolvedVocabularyEntry(definition.getVocabulary(), query, null, 0, subpath));
					} 
					// Exactly one matching candidate found
					else if (veCandidates.size()==1) {
						String resolvedKey = veCandidates.get(0).getKey();
						log.debug("Resolving '{}' against vocabulary '{}': One match found => set", query, definition.getVocabulary());
						resolved.add(new ResolvedVocabularyEntry(definition.getVocabulary(), query, resolvedKey, 1, subpath));
						TextPropertyValue.class.cast(v).setValue(resolvedKey);
					} 
					// Multiple matching candidates found => Using first, if probable match
					else {
						// TODO: This is not a rule that can stay, need something sophisticated here
						if (veCandidates.get(0).getScore()>0 && veCandidates.get(1).getScore()>0 && 
								veCandidates.get(0).getScore()>50 && veCandidates.get(0).getScore()/3>veCandidates.get(1).getScore()) {
							log.debug("Resolving '{}' against vocabulary '{}': {} matches found => most probable chosen", query, definition.getVocabulary(), veCandidates.size());
							resolved.add(new ResolvedVocabularyEntry(definition.getVocabulary(), query, veCandidates.get(0).getKey(), veCandidates.size(), subpath));
							TextPropertyValue.class.cast(v).setValue(veCandidates.get(0).getKey());
						} else {
							log.debug("Resolving '{}' against vocabulary '{}': {} matches found => unhandled", query, definition.getVocabulary(), veCandidates.size());
							resolved.add(new ResolvedVocabularyEntry(definition.getVocabulary(), query, null, veCandidates.size(), subpath));	
						}
					}
					if (!fromCache) {
						resolveCache.put(ResolveCacheKey.get(definition.getVocabulary(), query), veCandidates);
					}
					log.debug("{} candidates identified", veCandidates==null ? 0 : veCandidates.size());
				} catch (VocabularyLookupException e) {
					log.error("Failed to resolve vocabulary entry", e);
				}
			}
		}
	}
	
}
