package de.uniba.minf.registry.view.helper;

import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import de.uniba.minf.registry.model.BooleanPropertyValue;
import de.uniba.minf.registry.model.Property;
import de.uniba.minf.registry.model.PropertyImpl;
import de.uniba.minf.registry.model.PropertyList;
import de.uniba.minf.registry.model.PropertyValue;
import de.uniba.minf.registry.model.PropertyValueList;
import de.uniba.minf.registry.model.TextPropertyValue;
import de.uniba.minf.registry.model.definition.EntityDefinition;
import de.uniba.minf.registry.model.definition.HierarchicalPropertyDefinition;
import de.uniba.minf.registry.model.definition.PropertyDefinition;
import de.uniba.minf.registry.model.definition.PropertyDefinitionBlock;
import de.uniba.minf.registry.model.definition.SimplePropertyDefinition;
import de.uniba.minf.registry.model.definition.VocabularyPropertyDefinition;
import de.uniba.minf.registry.model.definition.SimplePropertyDefinition.SIMPLE_TYPES;
import de.uniba.minf.registry.model.entity.Entity;
import de.uniba.minf.registry.model.entity.EntityRelation;
import de.uniba.minf.registry.model.entity.EntityRelation.MainRelationTypes;
import de.uniba.minf.registry.model.vocabulary.VocabularyDefinition;
import de.uniba.minf.registry.repository.EntityDefinitionRepository;
import de.uniba.minf.registry.repository.EntityRepository;
import de.uniba.minf.registry.repository.VocabularyDefinitionRepository;
import lombok.AllArgsConstructor;
import lombok.Data;

@Component
public class EntityRelationsHelper {
	@Autowired private EntityRepository entityRepository;
	@Autowired private EntityDefinitionRepository entityDefinitionRepository;
	@Autowired private VocabularyDefinitionRepository vocabularyDefinitionRepository;
	
	private static final String RELATIONS_BLOCK = "model.generic.relations";
	private static final String RELATIONS_FIELDNAME_PREFIX = "~relations";
	
	private static final String RELATED_ENTITY_FIELDNAME = "~relatedEntity";
	private static final String RELATION_TYPE_FIELDNAME = "~relationType";
	private static final String RELATION_INVERSE_FIELDNAME = "~relationInverse";
	
	private static final String RELATED_ENTITY_MESSAGECODE = "model.generic.relations.related_entity";
	private static final String RELATION_TYPE_MESSAGECODE = "model.generic.relations.relation_type";
	private static final String RELATION_INVERSE_MESSAGECODE = "model.generic.relations.inverse";
	
	public void appendRelationsPropertyDefinitionBlock(EntityDefinition parentEntityDefinition) {
		parentEntityDefinition.getPropertyBlocks().add(this.getRelationsPropertyDefinitionBlock(parentEntityDefinition));
	}
	
	@Data
	class EntityRelationContainer {
		final EntityRelation relation;
		final Entity relatedEntity;
		final EntityDefinition relatedEntityDefinition;
		final boolean inverse;
		final String path;
	}
	
	private boolean isVocabularyEntryMatch(EntityRelationContainer erContainer, VocabularyPropertyDefinition vpd) {
		if (erContainer.getRelation().getTypeVocabulary()==null && vpd.getEntityRelationVocabulary()==null) {
			return true;
		}
		if (erContainer.getRelation().getTypeVocabulary()==null || vpd.getEntityRelationVocabulary()==null) {
			return false;
		} 
		if (erContainer.getRelation().getTypeVocabularyKey()==null && vpd.getEntityRelationType()==null) {
			return true;
		}
		if (erContainer.getRelation().getTypeVocabularyKey()==null || vpd.getEntityRelationType()==null) {
			return false;
		}
		
		return erContainer.getRelation().getTypeVocabulary().equals(vpd.getEntityRelationVocabulary()) && 
				erContainer.getRelation().getTypeVocabularyKey().equals(vpd.getEntityRelationType());
	}
		
	public void attachPropertiesForEntityRelations(Entity entity, EntityDefinition ed, boolean relationsToLatestOnly, Collection<EntityRelation> relations) {
		PropertyImpl pDef;
		
		Entity relEntity;
		EntityDefinition relEntityDef;

		// Collect required information on entity relations
		List<EntityRelationContainer> erContainers = new ArrayList<>(relations.size());
		for (EntityRelation er : relations) {
			relEntity = entityRepository.findById(er.getRelatedUniqueIdForEntity(entity)).orElse(null);
			if (relEntity==null || (relEntity.getNextVersionUniqueId()!=null && relationsToLatestOnly) ) {
				continue;
			}
			relEntityDef = entityDefinitionRepository.findCurrentByName(relEntity.getDefinitionName());
			erContainers.add(new EntityRelationContainer(er, relEntity, relEntityDef, er.isInverseForEntity(entity), er.getPathForEntity(entity)));
		}
		
		
		// Process explicitly defined relations
		List<EntityRelationContainer> processedContainers = new ArrayList<>();
		
		Map<String, List<String>> pathValueMap = new LinkedHashMap<>();
		for (VocabularyPropertyDefinition vpd : ed.getEntityVocabularyProperties()) {
			if (vpd.getName().startsWith(RELATED_ENTITY_FIELDNAME)) {
				continue;
			}
			
			for (EntityRelationContainer erContainer : erContainers) {
				if (vpd.getVocabulary().equals(erContainer.getRelatedEntityDefinition().getName())) {
					processedContainers.add(erContainer);
					if (this.isVocabularyEntryMatch(erContainer, vpd)) {
						List<String> values;
						if (pathValueMap.containsKey(erContainer.getPath())) {
							values = pathValueMap.get(erContainer.getPath());
						} else {
							values = new ArrayList<>();
							pathValueMap.put(erContainer.getPath(), values);
						}
						if (!values.contains(erContainer.relatedEntity.getEntityId())) {
							values.add(erContainer.relatedEntity.getEntityId());
						}						
					}
				}
			}
		}
		
		PropertyValueList lst;
		for (Entry<String, List<String>> pathValueEntry : pathValueMap.entrySet()) {
			if (pathValueEntry.getKey()==null) {
				continue;
			}
			if (pathValueEntry.getValue().size()==1) {
				entity.set(pathValueEntry.getKey(), pathValueEntry.getValue().get(0), true);
			} else {
				lst = new PropertyValueList();
				lst.addAll(pathValueEntry.getValue().stream().map(TextPropertyValue::new).toList());
				entity.set(pathValueEntry.getKey(), lst, true);
			}
		}
		
		
		erContainers.removeAll(processedContainers);
		
		
		// Process remaining relations
		Map<String, PropertyImpl> entityTypePropertyMap = new LinkedHashMap<>(); 
		for (EntityRelationContainer erContainer : erContainers) {
			if (entityTypePropertyMap.containsKey(erContainer.getRelatedEntityDefinition().getName())) {
				pDef = entityTypePropertyMap.get(erContainer.getRelatedEntityDefinition().getName());
			} else {
				pDef = new PropertyImpl();
				pDef.setLabel(RELATIONS_FIELDNAME_PREFIX + "_" + erContainer.getRelatedEntityDefinition().getName());
				pDef.setProperties(new ArrayList<>());
				entityTypePropertyMap.put(erContainer.getRelatedEntityDefinition().getName(), pDef);
			}
			pDef.getProperties().add(this.getPropertiesForEntityRelation(entity, erContainer.getRelatedEntity(), erContainer.isInverse(), erContainer.getRelation()));
		}
		
		
		for (Property p : entityTypePropertyMap.values()) {
			entity.add(p);
		}
	}
	
	public List<EntityRelation> getRelationsAndRemoveFromEntity(Entity e) {
		List<EntityRelation> relations = this.getExplicitRelationsAndRemoveFromEntity(e);
		relations.addAll(this.getImplicitRelationsAndRemoveFromEntity(e));
		
		return relations;
	}
	
	private List<EntityRelation> getImplicitRelationsAndRemoveFromEntity(Entity e) {
		List<Property> relationsProps = new ArrayList<>();
		for (Property p : e.getProperties()) {
			if (p.getLabel().startsWith(RELATIONS_FIELDNAME_PREFIX)) {
				relationsProps.add(p);
			}
		}
		
		List<EntityRelation> ers = this.createEntityRelations(e, relationsProps);
		
		
		// TODO Here implicit removed
		e.getProperties().removeAll(relationsProps);
				
		return ers;
	}
	
	private List<EntityRelation> getExplicitRelationsAndRemoveFromEntity(Entity e) {
		Map<String, RelationPropertyValuesContainer> explicitRelations = this.collectExplicitRelations(e.getProperties(), null);
		List<EntityRelation> ers = new ArrayList<>();
		if (explicitRelations.isEmpty()) {
			return ers;
		}
		for (Entry<String, RelationPropertyValuesContainer> entry : explicitRelations.entrySet()) {
			VocabularyPropertyDefinition vpd = entry.getValue().getDefinition().asVocabulary();
			entry.getValue().getValues().stream().forEach(v -> {
				EntityRelation er = this.getEntityRelation(e, v.asText(), vpd.getEntityRelationType(), vpd.getEntityRelationVocabulary(), false, null);
				if (er!=null) {
					er.setFromPath(entry.getKey());
					ers.add(er);
				}
			});
		}
		return ers;
	}
	
	private List<EntityRelation> createEntityRelations(Entity e, List<Property> relationsProperties) {		
		// Create relations
		List<EntityRelation> ers = new ArrayList<>();	
		for (Property p : relationsProperties) {
			// ~relations per entity table
			for (PropertyList pl : p.getProperties()) {
				EntityRelation er = this.getEntityRelationForProperties(e, pl.getProperties());
				if (er!=null) {
					ers.add(er);
				}
			}
		}
		return ers;
	}
	
	public PropertyList getPropertiesForEntityRelation(Entity entity, Entity relEntity, boolean inverse, EntityRelation er) {
		PropertyList pl = new PropertyList();
		pl.setProperties(new ArrayList<>());
				

		PropertyImpl p = new PropertyImpl();
		p.setLabel(RELATED_ENTITY_FIELDNAME);
		p.setValue(new TextPropertyValue(relEntity.getEntityId()));
		pl.add(p);
		
		p = new PropertyImpl();
		p.setLabel(RELATION_TYPE_FIELDNAME);
		p.setValue(new TextPropertyValue(er.getTypeVocabularyKey()));
		pl.add(p);
		
		p = new PropertyImpl();
		p.setLabel(RELATION_INVERSE_FIELDNAME);
		p.setValue(new BooleanPropertyValue(inverse));
		pl.add(p);

		return pl;
	}
	
	
	
	private boolean isExplicitRelation(PropertyDefinition vpd) {
		return vpd.isVocabulary() && vpd.asVocabulary().isEntity() && !vpd.getName().equals(RELATED_ENTITY_FIELDNAME);
	}
	
	
	@Data
	@AllArgsConstructor
	class RelationPropertyValuesContainer {
		private PropertyDefinition definition;
		List<PropertyValue> values;
	}
	
	private Map<String, RelationPropertyValuesContainer> collectExplicitRelations(List<Property> properties, String pathPrefix) {
		Map<String, RelationPropertyValuesContainer> relProps = new LinkedHashMap<>();
		
		List<Property> removeProps = new ArrayList<>();
		for (Property p : properties) {
			String path = pathPrefix==null ? p.getLabel() : pathPrefix + "." + p.getLabel();
			if (this.isExplicitRelation(p.getDefinition())) {
				List<PropertyValue> entityRelationValues = new ArrayList<>();
				for (PropertyValue pv : p.valuesAsList()) {
					if (TextPropertyValue.class.isAssignableFrom(pv.getClass()) && TextPropertyValue.class.cast(pv).isValidEntityId()) {
						entityRelationValues.add(pv);
					}
				}
				p.removeValues(entityRelationValues);
				
				// Remove whole property if no unresolved values remain 
				if (p.valuesAsList().isEmpty()) {
					removeProps.add(p);
				}
				
				// Collect relations
				if (!entityRelationValues.isEmpty()) {
					relProps.put(path, new RelationPropertyValuesContainer(p.getDefinition(), entityRelationValues));
				}				
			} else if (p.getDefinition().isHierarchical()) {
				relProps.putAll(this.collectExplicitRelationsForPropertyList(p.getProperties(), pathPrefix));
			}
		}
		
		// TODO Here implicit removed
		properties.removeAll(removeProps);
		return relProps;
	}
	
	private Map<String, RelationPropertyValuesContainer> collectExplicitRelationsForPropertyList(List<PropertyList> propertyLists, String pathPrefix) {
		Map<String, RelationPropertyValuesContainer> relProps = new LinkedHashMap<>();
		int i=0;
		for (PropertyList pl : propertyLists) {
			String path = pathPrefix + "[" + i++ + "]";
			relProps.putAll(this.collectExplicitRelations(pl.getProperties(), path));
		}
		return relProps;
	}

	public EntityRelation getEntityRelationForProperties(Entity saveEntity, List<Property> properties) {
		String entityId = null;
		String type = null;
		String vocabulary = null;
		boolean inverse = false;
		
		List<Property> addProperties = new ArrayList<>();
		
		for (Property p : properties) {
			if (p.getLabel().equals(RELATED_ENTITY_FIELDNAME)) {
				entityId = p.getValue().asText();
			} else if (p.getLabel().equals(RELATION_TYPE_FIELDNAME)) {
				type = p.getValue().asText();
				vocabulary = VocabularyPropertyDefinition.class.cast(p.getDefinition()).getVocabulary();
			} else if (p.getLabel().equals(RELATION_INVERSE_FIELDNAME)) {
				inverse = p.getValue().asBoolean();
			} else {
				addProperties.add(p);
			}
		}
		return this.getEntityRelation(saveEntity, entityId, type, vocabulary, inverse, addProperties);
	}
	
	private EntityRelation getEntityRelation(Entity saveEntity, String entityId, String type, String vocabulary, boolean inverse, List<Property> relationProperties) {
		Optional<Entity> relatedEntity = entityRepository.findLatestByEntityId(entityId);
		
		if (relatedEntity.isEmpty()) {
			return null;
		}
		
		EntityRelation er = new EntityRelation();		
		Entity fromEntity = inverse ? relatedEntity.get() : saveEntity;
		Entity toEntity = inverse ? saveEntity : relatedEntity.get();
	
		er.setFromUniqueId(fromEntity.getUniqueId());
		er.setToUniqueId(toEntity.getUniqueId());
		er.setFromEntityId(fromEntity.getEntityId());
		er.setToEntityId(toEntity.getEntityId());
		
		er.setMainRelationType(MainRelationTypes.CUSTOM);
		
		er.setTypeVocabulary(vocabulary);
		er.setTypeVocabularyKey(type);
		
		er.setProperties(relationProperties);
		
		return er;
	}
	
	public PropertyDefinitionBlock getRelationsPropertyDefinitionBlock(EntityDefinition parentEntityDefinition) {
		PropertyDefinitionBlock relationsMetadataBlock = new PropertyDefinitionBlock();
    	relationsMetadataBlock.setName(RELATIONS_BLOCK);
    	relationsMetadataBlock.setIdentifier(RELATIONS_BLOCK);
    	relationsMetadataBlock.setMessageCode(RELATIONS_BLOCK);
    	relationsMetadataBlock.setProperties(new ArrayList<>());
    
    	List<String> explicitRelations = parentEntityDefinition.getEntityVocabularyProperties().stream().filter(this::isExplicitRelation).map(pd -> pd.getName()).toList();
    	
    	for (String relatedDefName : entityDefinitionRepository.findAllDefinitionNames(false)) {
			// If relation is explicitly defined, skip here
    		if (explicitRelations.contains(relatedDefName)) {
				continue;
			}
	    	   	
	    	List<String> multiList = new ArrayList<>(1);
	    	multiList.add("true");
	    	
	    	HierarchicalPropertyDefinition relationItemDefinition = new HierarchicalPropertyDefinition();
	    	relationItemDefinition.setName(RELATIONS_FIELDNAME_PREFIX + "_" + relatedDefName);
	    	relationItemDefinition.setIdentifier(parentEntityDefinition.getName() + "." + relationItemDefinition.getName());
	    	relationItemDefinition.setMessageCode("~" + relatedDefName);
	    	relationItemDefinition.setProperties(new ArrayList<>());
	    	relationItemDefinition.setMultiplicity(multiList);
	    	
	    	VocabularyPropertyDefinition adminMetadataItemDefinition = new VocabularyPropertyDefinition();
	    	//adminMetadataItemDefinition.setMessageCode(code);
	    	adminMetadataItemDefinition.setName(RELATED_ENTITY_FIELDNAME);
	    	adminMetadataItemDefinition.setIdentifier(parentEntityDefinition.getName() + "." + adminMetadataItemDefinition.getName());
	    	adminMetadataItemDefinition.setMessageCode(RELATED_ENTITY_MESSAGECODE);
	    	adminMetadataItemDefinition.setVocabulary(relatedDefName);
	    	adminMetadataItemDefinition.setQuery(true);
	    	adminMetadataItemDefinition.setStrict(true);
	    	adminMetadataItemDefinition.setEntity(true);
	    	adminMetadataItemDefinition.setMandatory(true);
	    	
	    	relationItemDefinition.getProperties().add(adminMetadataItemDefinition);
	    	
	    	
	    	VocabularyDefinition typeVocabularyDefinition = vocabularyDefinitionRepository.findCurrentByName("_relations_" + parentEntityDefinition.getName() + "_" + relatedDefName);
	    	if (typeVocabularyDefinition==null) {
	    		typeVocabularyDefinition = vocabularyDefinitionRepository.findCurrentByName("_relations_" + relatedDefName + "_" + parentEntityDefinition.getName());
	    	}
	    	if (typeVocabularyDefinition==null) {
	    		typeVocabularyDefinition = vocabularyDefinitionRepository.findCurrentByName("_relations_" + relatedDefName);
	    	}
	    	if (typeVocabularyDefinition==null) {
	    		typeVocabularyDefinition = vocabularyDefinitionRepository.findCurrentByName("_relations");
	    	}
	    	if (typeVocabularyDefinition!=null) {
	    	
	    		adminMetadataItemDefinition = new VocabularyPropertyDefinition();
		    	//adminMetadataItemDefinition.setMessageCode(code);
		    	adminMetadataItemDefinition.setName(RELATION_TYPE_FIELDNAME);
		    	adminMetadataItemDefinition.setMessageCode(RELATION_TYPE_MESSAGECODE);
		    	adminMetadataItemDefinition.setIdentifier(parentEntityDefinition.getName() + "." + adminMetadataItemDefinition.getName());
		    	adminMetadataItemDefinition.setVocabulary(typeVocabularyDefinition.getName());
		    	adminMetadataItemDefinition.setStrict(true);
		    	adminMetadataItemDefinition.setMandatory(true);
		    	
		    	relationItemDefinition.getProperties().add(adminMetadataItemDefinition);
	    	}

	    	SimplePropertyDefinition adminMetadataItemDefinition1 = new SimplePropertyDefinition();
	    	//adminMetadataItemDefinition.setMessageCode(code);
	    	adminMetadataItemDefinition1.setType(SIMPLE_TYPES.BOOLEAN);
	    	adminMetadataItemDefinition1.setName(RELATION_INVERSE_FIELDNAME);
	    	adminMetadataItemDefinition1.setMessageCode(RELATION_INVERSE_MESSAGECODE);
	    	adminMetadataItemDefinition1.setIdentifier(parentEntityDefinition.getName() + "." + adminMetadataItemDefinition1.getName());
	    	relationItemDefinition.getProperties().add(adminMetadataItemDefinition1);
	    	
	    	relationsMetadataBlock.getProperties().add(relationItemDefinition);
		}
		return relationsMetadataBlock;
	}
}
