package de.uniba.minf.registry.view.controller;

import java.net.MalformedURLException;
import java.net.URL;
import java.time.format.DateTimeFormatter;
import java.time.format.FormatStyle;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Locale;
import java.util.Optional;
import java.util.TimeZone;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.server.ResponseStatusException;

import de.uniba.minf.registry.model.IntegerPropertyValue;
import de.uniba.minf.registry.model.PersistedUser;
import de.uniba.minf.registry.model.Property;
import de.uniba.minf.registry.model.PropertyImpl;
import de.uniba.minf.registry.model.PropertyList;
import de.uniba.minf.registry.model.PropertyValue;
import de.uniba.minf.registry.model.TextPropertyValue;
import de.uniba.minf.registry.model.definition.EntityDefinition;
import de.uniba.minf.registry.model.definition.PropertyDefinition;
import de.uniba.minf.registry.model.definition.SimplePropertyDefinition;
import de.uniba.minf.registry.model.definition.SimplePropertyDefinition.SIMPLE_TYPES;
import de.uniba.minf.registry.model.entity.Entity;
import de.uniba.minf.registry.repository.EntityRepository;
import de.uniba.minf.registry.repository.UserRepository;
import de.uniba.minf.registry.service.EntityService;
import de.uniba.minf.registry.view.helper.PropertyViewItemCombiner;
import de.uniba.minf.registry.view.model.PropertyBlockViewItem;
import de.uniba.minf.registry.view.model.PropertyViewItem;
import de.uniba.minf.registry.view.model.SimplePropertyViewItem;
import lombok.extern.slf4j.Slf4j;

@Slf4j
@Controller
@RequestMapping("/entity")
public class EntityViewController extends BaseViewController {
	
	@Autowired private EntityRepository entityRepository;
	@Autowired private EntityService entityService;
	
	@Autowired private PropertyViewItemCombiner propertyViewItemCombiner;
	
	@Autowired private UserRepository userRepository;
	
	public EntityViewController() {
		super("entity");
	}
		
	@GetMapping("/new/{entityDefinitionName}/")
	public String newEntity(@PathVariable("entityDefinitionName") String entityDefinitionName, Model model, Locale locale) {
		EntityDefinition entityDefinition = entityDefinitionService.findCurrentByName(entityDefinitionName, false);
		
		Entity e = new Entity();
		e.setDefinitionName(entityDefinition.getName());
		e.setDefinitionVersion(entityDefinition.getVersion());
		
		return this.editEntity(e, entityDefinition, model, locale);
	}
	
	@GetMapping("{uniqueId}/")
	public String showEntity(@PathVariable("uniqueId") String uniqueId, Model model, Locale locale) {
		Optional<Entity> entity = entityService.findById(uniqueId, true);
		if (!entity.isPresent()) {
			entity = entityService.findLatestByEntityId(uniqueId, true);
		}
		if (!entity.isPresent()) {
			throw new ResponseStatusException(HttpStatus.NOT_FOUND, "~entity not found"); 
	    	}
	
	    	EntityDefinition entityDefinition = entityDefinitionService.findCurrentByName(entity.get().getDefinitionName(), false);
	    
	    	return this.editEntity(entity.get(), entityDefinition, model, locale);
	    }
	   
	public String editEntity(Entity entity, EntityDefinition entityDefinition, Model model, Locale locale) {
	    model.addAttribute("entity", entity);
		model.addAttribute("def", entityDefinition);
		    	    	
		List<PropertyBlockViewItem> propViewItems = propertyViewItemCombiner.getFromDefinitionBlocksAndEntity(entityDefinition.getPropertyBlocks(), entity.getProperties());
		propViewItems.add(this.getMetadataBlock(entity, locale));
		//propViewItems.add(this.getRelationsBlock(entity, locale));
		model.addAttribute("propBlockViewItems", propViewItems);
		
		return "entities/edit";
	}
	    
	@GetMapping("/includes/{propertyIdentifier}")
	public String showEntityInclude(@PathVariable("propertyIdentifier") String propertyIdentifier, @RequestParam("path") String path, Model model, Locale locale) {
		
		String[] propertyPath = propertyIdentifier.split("\\.");
		
		EntityDefinition entityDefinition = entityDefinitionService.findCurrentByName(propertyPath[0], false);
		
		PropertyDefinition propertyDefinition = entityDefinition.getDefinition(Arrays.copyOfRange(propertyPath, 1, propertyPath.length));
		
		PropertyList pl = new PropertyList();
		Property p = new PropertyImpl();
		
		p.setProperties(new ArrayList<>());
		p.getProperties().add(pl);
		
		PropertyViewItem propViewItem = propertyViewItemCombiner.getFromDefinitionAndEntity(propertyDefinition, p);
		
		
		model.addAttribute("propViewItem", propViewItem);
		model.addAttribute("path", path);
		
		// Required for its readOnly==false
		model.addAttribute("entity", new Entity());
	
		return "entities/async_property";
	}
	
	private PropertyBlockViewItem getMetadataBlock(Entity entity, Locale locale) {
		DateTimeFormatter formatter = DateTimeFormatter
		        .ofLocalizedDateTime(FormatStyle.LONG, FormatStyle.MEDIUM)
		        .withLocale(locale)
		        .withZone(TimeZone.getTimeZone("Europe/Berlin").toZoneId());
		 
		    	
		PropertyBlockViewItem adminMetadataBlock = new PropertyBlockViewItem();
		adminMetadataBlock.setIdentifier("model.generic.metadata");
		adminMetadataBlock.setMessageCode("model.generic.metadata");
		
		List<PropertyViewItem> adminMetadataItems = new ArrayList<>();
		
		URL versionURL = null;
		URL entityURL = null;
		try {
			versionURL = new URL(baseUrl + "entity/" + entity.getUniqueId() + "/");
			entityURL = new URL(baseUrl + "entity/" + entity.getEntityId() + "/");
		} catch (MalformedURLException e) {
			log.error("Failed to build entity URLs", e);
		}	
		
		adminMetadataItems.add(this.getMetadataItem("entity_id", new TextPropertyValue(entity.getEntityId(), null, entityURL.toString())));
		adminMetadataItems.add(this.getMetadataItem("unique_id", new TextPropertyValue(entity.getUniqueId(), null, versionURL.toString())));
		
		if (entity.getNextVersionUniqueId()!=null) {
			adminMetadataItems.add(this.getMetadataItem("next_unique_id", new TextPropertyValue(entity.getNextVersionUniqueId(), null, entityURL.toString())));
		}
		if (entity.getCreationInstant()!=null) {
			adminMetadataItems.add(this.getMetadataItem("timestamp", new TextPropertyValue(formatter.format(entity.getCreationInstant()))));
		}
		
		Optional<PersistedUser> user = Optional.empty();
		if (entity.getUserUniqueId()!=null) {
			user = userRepository.findById(entity.getUserUniqueId());
			if (user.isPresent()) {
				adminMetadataItems.add(this.getMetadataItem("user", new TextPropertyValue(user.get().getUsername())));
			}
		}
		
		if (entity.getEntityId()!=null) {
			
			int versions = (int)entityRepository.countVersionsByEntityId(entity.getEntityId());
			Entity initialEntity = entityRepository.findEarliestByEntityId(entity.getEntityId()).get();
			
			adminMetadataItems.add(this.getMetadataItem("version_count", new IntegerPropertyValue(versions)));
		
			if (initialEntity.getCreationInstant()!=null) {
				adminMetadataItems.add(this.getMetadataItem("created_timestamp", new TextPropertyValue(formatter.format(initialEntity.getCreationInstant()))));
			}
			
			if (initialEntity.getUserUniqueId()!=null) {
				if (entity.getUserUniqueId()==null || !initialEntity.getUserUniqueId().equals(entity.getUserUniqueId())) {
					user = userRepository.findById(initialEntity.getUserUniqueId());
				}
				if (user.isPresent()) {
					adminMetadataItems.add(this.getMetadataItem("initial_user", new TextPropertyValue(user.get().getUsername())));
				}
			}
		}
		
		adminMetadataBlock.setPropertyViewItems(adminMetadataItems);
		
		return adminMetadataBlock;
	}    
	
	private SimplePropertyViewItem getMetadataItem(String label, PropertyValue value) {
		String code = "model.generic.metadata." + label;
		SimplePropertyDefinition adminMetadataItemDefinition = new SimplePropertyDefinition();
		adminMetadataItemDefinition.setMessageCode(code);
		adminMetadataItemDefinition.setName(label);
		adminMetadataItemDefinition.setIdentifier(code);
		adminMetadataItemDefinition.setType(SIMPLE_TYPES.TEXT);
		
		Property property = new PropertyImpl();
		property.setLabel(code);
		property.setValue(value);
		
		SimplePropertyViewItem adminMetadataItem = new SimplePropertyViewItem(adminMetadataItemDefinition, property); 
		adminMetadataItem.setReadonly(true);
		
		return adminMetadataItem;
	}
	
}
