package de.uniba.minf.registry.service;

import java.io.IOException;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;

import org.apache.tika.Tika;
import org.javers.core.Javers;
import org.javers.core.diff.Diff;
import org.javers.core.diff.DiffBuilder;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.stereotype.Component;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ObjectNode;

import am.ik.yavi.core.ConstraintViolations;
import am.ik.yavi.core.Validator;
import de.uniba.minf.registry.model.BasePropertyValue;
import de.uniba.minf.registry.model.Property;
import de.uniba.minf.registry.model.PropertyList;
import de.uniba.minf.registry.model.PropertyValue;
import de.uniba.minf.registry.model.TextPropertyValue;
import de.uniba.minf.registry.model.definition.AutofillPropertyDefinition;
import de.uniba.minf.registry.model.definition.EntityDefinition;
import de.uniba.minf.registry.model.definition.PropertyDefinition;
import de.uniba.minf.registry.model.entity.AutoqueryEntityLookupService;
import de.uniba.minf.registry.model.entity.Entity;
import de.uniba.minf.registry.model.entity.EntityRelation;
import de.uniba.minf.registry.model.serialization.EntityDeserializer;
import de.uniba.minf.registry.model.validation.EntityValidator;
import de.uniba.minf.registry.model.validation.exception.ValidationConfigurationException;
import de.uniba.minf.registry.model.vocabulary.ValidationEntityService;
import de.uniba.minf.registry.model.vocabulary.VocabularyDefinition;
import de.uniba.minf.registry.model.vocabulary.VocabularyEntry;
import de.uniba.minf.registry.model.vocabulary.VocabularyLookupException;
import de.uniba.minf.registry.model.vocabulary.VocabularyLookupService;
import de.uniba.minf.registry.repository.EntityDefinitionRepository;
import de.uniba.minf.registry.repository.EntityRelationRepository;
import de.uniba.minf.registry.repository.EntityRepository;
import de.uniba.minf.registry.repository.VocabularyDefinitionRepository;
import de.uniba.minf.registry.view.helper.EntityRelationsHelper;
import de.uniba.minf.registry.view.helper.PropertyDefinitionHelper;
import lombok.extern.slf4j.Slf4j;

@Slf4j
@Component
public class EntityServiceImpl implements EntityService {
	
	@Autowired private EntityRepository entityRepo;
	@Autowired private EntityRelationRepository entityRelationRepo;
	@Autowired private EntityDefinitionService entityDefinitionService;
	@Autowired private EntityDefinitionRepository entityDefRepo;
	@Autowired private PropertyDefinitionHelper propertyDefinitionHelper;
	@Autowired private VocabularyLookupService vocabularyLookupService;
	@Autowired private VocabularyDefinitionRepository vocabularyDefinitionRepo;
	@Autowired private EntityRelationsHelper entityRelationsHelper;
	@Autowired private AutoqueryEntityLookupService entityLookupService;
	
	@Autowired private Tika tika;
	
	@Autowired @Qualifier("yamlMapper") private ObjectMapper yamlMapper;
	@Autowired @Qualifier("jsonMapper") private ObjectMapper jsonMapper;
		
	@Autowired private Javers javers;
	
	@Override
	public Diff getChanges(Entity e) {
		Optional<Entity> eLatest = findLatestByEntityId(e.getEntityId(), true);
		if (eLatest.isEmpty()) {
			return DiffBuilder.empty();
		}
		if (e.getPropertyDefinitions()!=null && !e.getPropertyDefinitions().isEmpty()) {
			EntityDefinition ed = entityDefRepo.findCurrentByName(eLatest.get().getDefinitionName());
			propertyDefinitionHelper.mergeWithDefinition(eLatest.get(), ed, true);
		}
		return javers.compare(e.getProperties(), eLatest.get().getProperties());
	}
	
	@Override
	public Optional<Entity> findById(String uniqueId, boolean loadRelations) {
		Optional<Entity> e = entityRepo.findById(uniqueId);
		if (e.isPresent() && loadRelations) {
			this.loadRelations(e.get());
		}
		return e;
	}
	
	@Override
	public Optional<Entity> findLatestByEntityId(String entityId) {
		return this.findLatestByEntityId(entityId, false);
	}
	
	@Override
	public Optional<Entity> findLatestByEntityId(String entityId, boolean loadRelations) {
		Optional<Entity> e = entityRepo.findLatestByEntityId(entityId);
		if (e.isPresent() && loadRelations) {			
			this.loadRelations(e.get());
		}
		return e;
	}
	
	@Override
	public void loadRelations(Entity entity) {
		Collection<EntityRelation> relations = entityRelationRepo.findAllByEntityUniqueId(entity.getUniqueId());
		if (relations.isEmpty()) {
			return;
		}
		EntityDefinition ed = entityDefRepo.findCurrentByName(entity.getDefinitionName());
		if (entity.getPropertyDefinitions()==null) {
			propertyDefinitionHelper.mergeWithDefinition(entity, ed, false);
		}
		entityRelationsHelper.attachPropertiesForEntityRelations(entity, ed, true, relations);
	}
	
	@Override
	public Entity prepareEntity(String entityId, String definition, boolean copyAsTemplate, boolean changeToDraft, boolean changeToPublished) {
		Entity e = new Entity();
		
		// New template => new entityId 
		if (!copyAsTemplate) {
			e.setEntityId(entityId);
		}
		
		Optional<Entity> eExist = !copyAsTemplate && entityId!=null ? entityRepo.findLatestByEntityId(entityId) : Optional.empty();
		if (eExist.isPresent()) {
			e.setDraft(eExist.get().isDraft());
			e.setDefinitionName(eExist.get().getDefinitionName());
			e.setTemplate(eExist.get().isTemplate());
		}  else {
			e.setDefinitionName(definition);
		}

		// Set as template if new template or already a template
		if (copyAsTemplate || e.isTemplate()) {
			e.setTemplate(true);
			e.setDraft(false);
		} else {
			// Draft if not published AND (has been draft OR is now saved as draft)
			e.setDraft(!changeToPublished && (e.isDraft() || changeToDraft));
			e.setTemplate(false);
		}
		return e;
	}
	
	@Override
	public Entity fillEntityFromDataArray(Entity e, JsonNode formData) {
		String name;
		String value;
		String prop;
		TextPropertyValue propValue;
		
		Map<String, TextPropertyValue> valueMap = new HashMap<>();
		
		for (JsonNode fieldNode : formData) {
			name = fieldNode.get("name").asText();
			value = fieldNode.get("value").asText();
			prop = null;
			// Skip system properties
			if (value.isEmpty() || name.startsWith("_")) {
				continue;
			}

			if (name.indexOf('#')>=0) {
				prop = name.substring(name.indexOf('#')+1);
				name = name.substring(0, name.indexOf('#'));
			}
			if (valueMap.containsKey(name)) {
				propValue = valueMap.get(name);
			} else {
				propValue = new TextPropertyValue();
				valueMap.put(name, propValue);
			}
			if (prop!=null) {
				if (prop.equals("lang")) {
					propValue.setLang(value);
				}
			} else {
				propValue.setValue(value);
			}
		}
		
		for (Entry<String, TextPropertyValue> v : valueMap.entrySet()) {
			e.set(v.getKey(), v.getValue());
		}
		
		
		return e;
	}
	
	@Override
	public List<Entity> readEntitiesFromJson(String input, String entityDefinition) throws JsonProcessingException, IllegalArgumentException {
		JsonNode tree = jsonMapper.readTree(input);
		return this.readEntities(jsonMapper, tree, entityDefinition);
	}
	
	@Override
	public List<Entity> readEntitiesFromYaml(String input, String entityDefinition) throws JsonProcessingException, IllegalArgumentException {
		JsonNode tree = yamlMapper.readTree(input);
		return this.readEntities(yamlMapper, tree, entityDefinition);
	}
	
	@Override
	public List<Entity> readEntitiesFromURL(URL url, String entityDefinition) throws IllegalArgumentException, IOException {
		ObjectMapper mapper;
		String mimeType = tika.detect(url);
		if (mimeType!=null && (mimeType.endsWith("/x-yaml") || mimeType.endsWith("/yaml"))) {
			mapper = yamlMapper;
		} else if (mimeType!=null && mimeType.endsWith("/json")) {
			mapper = jsonMapper;
		} else {
			throw new IllegalArgumentException("Entity deserialization currently only supports JSON and XAML. Detected type: " + mimeType);
		}
		
		JsonNode tree = mapper.readTree(url);
		return this.readEntities(mapper, tree, entityDefinition);
	}
		
	@Override
	public List<ConstraintViolations> validateEntities(EntityDefinition ed, List<Entity> entities) throws ValidationConfigurationException {
		return this.validateEntities(ed, entities, false);
	}
	
	@Override
	public List<ConstraintViolations> validateEntities(EntityDefinition ed, List<Entity> entities, boolean skipCompleteness) throws ValidationConfigurationException {
		Validator<Entity> ev = new EntityValidator(ed, vocabularyLookupService, entityLookupService, ValidationEntityService.class.cast(this), skipCompleteness).build();
		List<ConstraintViolations> result = new ArrayList<>();
		for (Entity e : entities) {
			result.add(this.validateEntity(ev, e));
		}
		return result;
	}
	
	@Override
	public ConstraintViolations validateEntity(EntityDefinition ed, Entity e) throws ValidationConfigurationException {
		// Skip completeness checks if entity is a template
		Validator<Entity> ev = new EntityValidator(ed, vocabularyLookupService, entityLookupService, ValidationEntityService.class.cast(this), e.isTemplate()).build();
		return this.validateEntity(ev, e);
	}
	
	@Override
	public ConstraintViolations validateEntity(Validator<Entity> ev, Entity e) throws ValidationConfigurationException {
		ConstraintViolations violations = ev.validate(e);
		if (log.isDebugEnabled() && !violations.isValid()) {
			StringBuilder validationErrorBuilder = new StringBuilder();
			violations.forEach(x -> validationErrorBuilder.append(String.format("[%s],[%s] => %s\n", x.name(), x.messageKey(), x.message())));
			log.debug("Entity failed validation: \n{}", validationErrorBuilder.toString());
		}
		return violations;
	}

	@Override
	public Entity save(Entity e) {
		List<EntityRelation> ers = this.getAndDetachEntityRelationProperties(e);
		entityRepo.save(e);
		
		for (EntityRelation er : ers) {
			if (er.getFromEntityId()==null || er.getFromEntityId().equals(e.getEntityId())) {
				er.setFromUniqueId(e.getUniqueId());
				er.setFromEntityId(e.getEntityId());
			}
			if (er.getToEntityId()==null || er.getToEntityId().equals(e.getEntityId())) {
				er.setToUniqueId(e.getUniqueId());
				er.setToEntityId(e.getEntityId());
			}
		}
		
		entityRelationRepo.saveAll(ers);
		return e;
	}
	
	private List<EntityRelation> getAndDetachEntityRelationProperties(Entity e) {
		if (e.getProperties()==null) {
			return new ArrayList<>();
		}
		
		if (e.getPropertyDefinitions()==null) {
			EntityDefinition ed = entityDefinitionService.findCurrentByName(e.getDefinitionName(), true);
			propertyDefinitionHelper.mergeWithDefinition(e, ed, true);
		}
		return entityRelationsHelper.getRelationsAndRemoveFromEntity(e);
	}
	
		
	@Override
	public void setOrCreateRelatedEntities(Entity e, EntityDefinition ed) {
		// For all entity vocabulary properties -> iterate all values and processs
		ed.getEntityVocabularyProperties().stream().forEach(evpd -> {
			e.get(evpd).stream().flatMap(p -> p.valuesAsList().stream()).forEach(pv -> {
				// Relates already to entityId -> nothing to do
				if (entityRepo.findLatestByEntityId(pv.asText()).isEmpty()) {
					EntityDefinition relEd = entityDefRepo.findCurrentByName(evpd.getVocabulary());
					if (relEd==null) {
						log.warn("Queried unknown entity definition '{}'", evpd.getVocabulary());
						return;
					}
					try {
						if (entityLookupService.canResolveId(evpd.getVocabulary(), pv.asText())) {
							Optional<Entity> matchingIdEntity = this.findByExternalIdentifier(relEd, pv.asText()).stream().findFirst();
							Entity relE;
							// Entity found based on external identifiers
							if (matchingIdEntity.isPresent()) {
								relE = matchingIdEntity.get();
							} else {
								if (!relEd.hasAutoqueryProperties()) {
									log.warn("Entity definition '{}' has no configured autoquery properties", evpd.getVocabulary());
									return;
								}
								relE = this.createByAutoqueryAndAutopopulate(relEd, pv.asText());
								
								// Save as new entity and relate e to entityId
								entityRepo.save(relE);
							}
							// Vocabulary properties are always text -> safe to cast
							TextPropertyValue.class.cast(pv).setValue(relE.getEntityId());
						}
					} catch (VocabularyLookupException ex) {
						log.error("Failed trying to resolve vocabulary entry", ex);
					}
				}
			});
		});
	}
	
	@Override
	public Collection<Entity> findByExternalIdentifier(EntityDefinition ed, String identifier) {
		List<PropertyDefinition> idProps = ed.getExternalIdentifierProperties();
		if (idProps.isEmpty()) {
			return new ArrayList<>(0);
		}
		Criteria[] identifierCriteria = new Criteria[idProps.size()];
		for (int i=0; i<idProps.size(); i++) {
			String[] pathWithEntity = idProps.get(i).getIdentifier().split("\\.");
			Criteria c = null;
			// Create query criteria from the deepest nesting level
			for (int j=pathWithEntity.length-1; j>=1; j--) {
				// Match label and embedded value
				if (j==pathWithEntity.length-1) {
					c = new Criteria().andOperator(
							Criteria.where("label").is(pathWithEntity[j]),
							new Criteria().orOperator(
									Criteria.where("value.value").is(identifier),
									Criteria.where("values").elemMatch(Criteria.where("value").is(identifier))
							));
				} else {
					// Wrap hierarchy
					c = Criteria.where("label").is(pathWithEntity[j])
							.and("properties").elemMatch(
									Criteria.where("properties").elemMatch(c)
									);
				}
				
			}

			// Entity properties
			c = Criteria.where("properties").elemMatch(c);
			identifierCriteria[i] = c;
		}
		
		
		Criteria cr = new Criteria().andOperator(Criteria.where("definitionName").is(ed.getName()), new Criteria().orOperator(identifierCriteria));
		
		return entityRepo.findLatestByCriteria(cr);
	}
	
	@Override
	public Collection<Entity> findByExternalIdentifier(String definition, String identifier) {
		EntityDefinition ed = entityDefRepo.findCurrentByName(definition);
		if (ed==null) {
			return new ArrayList<>(0);
		}
		return this.findByExternalIdentifier(ed, identifier);
	}

	private Entity createByAutoqueryAndAutopopulate(EntityDefinition ed, String key) {
		Entity e = new Entity();
		ed.getAutoqueryProperties().stream().forEach(aqPropertyDefinition -> {
			String path = aqPropertyDefinition.getIdentifier().substring(aqPropertyDefinition.getIdentifier().indexOf('.')+1);
			e.set(path, key);
		});
		propertyDefinitionHelper.mergeWithDefinition(e, ed, false);
		this.autopopulateVocabularyData(e, ed);
		return e;
	}
	
	@Override
	public List<Entity> createEntitiesByAutoquery(String definition, String query) {
		List<Entity> result = new ArrayList<>();
		try {
			List<VocabularyEntry> ves = entityLookupService.search(definition, query);
			EntityDefinition ed = entityDefRepo.findCurrentByName(definition);
			if (ed==null) {
				log.warn("Queried unknown entity definition '{}'", definition);
				return result;
			}
			if (!ed.hasAutoqueryProperties()) {
				log.warn("Entity definition '{}' has no configured autoquery properties", definition);
				return result;
			}
			
			ves.stream().forEach(ve -> {
				Entity e = this.createByAutoqueryAndAutopopulate(ed, ve.getKey());
				e.setEntityId(ve.getKey());
				result.add(e);
			});
		} catch (VocabularyLookupException e) {
			log.error("Failed to autoquery entities", e);
		}
		return result;
	}
	
	@Override
	public void autopopulateVocabularyData(Entity e, EntityDefinition ed) {
		ed.getAutofillProperties().stream().forEach(afPropertyDefinition -> {
			String[] path = afPropertyDefinition.getIdentifier().split("\\.");
			Property vocabularyProp = e.get(Arrays.copyOfRange(path, 1, path.length));
			if (!vocabularyProp.isMissing()) {
				vocabularyProp.valuesAsList().stream().forEach(vocabularyPropVal -> {
					try {
						VocabularyEntry ve = vocabularyLookupService.resolve(afPropertyDefinition.getVocabulary(), vocabularyPropVal.asText());
						if (ve!=null) {
							VocabularyDefinition vd = vocabularyDefinitionRepo.findCurrentByName(afPropertyDefinition.getVocabulary());
							propertyDefinitionHelper.mergeWithDefinition(ve, vd, true);
							mergeAutofillProperties(e, ve.getProperties(), null, afPropertyDefinition.isAutofillAll() ? null : afPropertyDefinition.getAutofillProperties(), vd.getName());
						}
						
					} catch (VocabularyLookupException e1) {
						log.error("Failed to resolve autofill property", e);
					}
				});
			}
		});
	}
	
	@Override
	public void applyValueMappings(List<Entity> entities, Map<String, String> valueMap) {
		entities.stream().forEach(e -> this.applyValueMappingsToProperties(e.getProperties(), valueMap));
	}

	@Override
	public void applyValueMappings(Entity entity, Map<String, String> valueMap) {
		this.applyValueMappingsToProperties(entity.getProperties(), valueMap);
	}
	
	private void applyValueMappingsToProperties(List<Property> properties, Map<String, String> valueMap) {
		properties.stream().forEach(p -> {
			p.valuesAsList().stream()
				.filter(v -> TextPropertyValue.class.isAssignableFrom(v.getClass()) && valueMap.containsKey(v.asText()))
				.forEach(
						v -> TextPropertyValue.class.cast(v).setValue(valueMap.get(v.asText()))
				);
			
			if (p.getProperties()!=null) {
				this.applyValueMappingsToPropertyLists(p.getProperties(), valueMap);
			}
		});
	}
	
	private void applyValueMappingsToPropertyLists(List<PropertyList> propertyLists, Map<String, String> valueMap) {
		propertyLists.stream().forEach(pl -> this.applyValueMappingsToProperties(pl.getProperties(), valueMap));
	}
		
	private List<Entity> readEntities(ObjectMapper mapper, JsonNode tree, String entityDefinition) throws JsonProcessingException, IllegalArgumentException {
		List<Entity> entities = new ArrayList<>();

		// Characteristic for transformation service response
		if (!tree.isArray() && tree.has("_links") && (tree.has("items") || tree.has("item"))) {
			tree = tree.has("items") ? tree.get("items") : tree.get("item");
		}
		if (tree.isArray()) {
			for (JsonNode n : tree) {
				entities.add(this.readEntity(mapper, n, entityDefinition));
			}
		} else {
			entities.add(this.readEntity(mapper, tree, entityDefinition));
		}
		return entities;
	}
		
	private Entity readEntity(ObjectMapper mapper, JsonNode n, String entityDefinition) throws JsonProcessingException, IllegalArgumentException {
		// Prefixed entity object
		if (n.has(entityDefinition) && n.size()==1) {
			n = n.get(entityDefinition);
		}
		if (n.isObject() && !n.has(EntityDeserializer.ENTITY_FIELD)) {
			ObjectNode.class.cast(n).put(EntityDeserializer.ENTITY_FIELD, entityDefinition);
		}
		return mapper.treeToValue(n, Entity.class);
	}
	
	private void mergeAutofillProperties(Entity target, List<Property> properties, String pathPrefix, List<AutofillPropertyDefinition> propertyFilter, String vocabularyName) {
		/*
		 * This does not work when renaming in hierarchical scenarios, kick or improve
		 */
		for (Property mergeP : properties) {
			if (mergeP.getDefinition()!=null) {
				// Remove vocabulary prefix from identifier for filtering
				String filterIdentifier = mergeP.getDefinition().getIdentifier().substring(mergeP.getDefinition().getIdentifier().indexOf('.')+1); 
				if (mergeP.getDefinition().isHierarchical()) {
					String subPath = pathPrefix==null ? mergeP.getLabel() : pathPrefix + "." + mergeP.getLabel();
					this.mergeAutofillPropertyLists(target, mergeP.getProperties(), subPath, propertyFilter, vocabularyName);
				} else { 
					String subPath;
					if (propertyFilter==null) {
						subPath = pathPrefix==null ? mergeP.getLabel() : pathPrefix + "." + mergeP.getLabel();
					} else if (propertyFilter.stream().anyMatch(p -> p.filterApplies(filterIdentifier) )) {
						AutofillPropertyDefinition propDef = propertyFilter.stream().filter(p -> p.filterApplies(filterIdentifier)).findFirst().orElse(null);
						if (propDef.getExternalName()!=null) {
							subPath = pathPrefix==null ? propDef.getPropertyName() : pathPrefix + "." + propDef.getPropertyName();
						} else {
							subPath = pathPrefix==null ? mergeP.getLabel() : pathPrefix + "." + mergeP.getLabel();
						}
					} else {
						continue;
					}
					PropertyValue p = mergeP.getValue();
					if (p.isMultivalue()) {
						p.valuesAsList().stream().forEach(v -> BasePropertyValue.class.cast(v).setSource(vocabularyName));
					} else {
						BasePropertyValue.class.cast(p).setSource(vocabularyName);
					}
					target.set(subPath, mergeP.getValue());
				}
			}
		}
	}
	
	private void mergeAutofillPropertyLists(Entity target, List<PropertyList> propertyLists, String pathPrefix, List<AutofillPropertyDefinition> propertyFilter, String vocabularyName) {
		// This sets entity properties based on vocabulary indices and might overwrite existing data
		//  ignoring this for now, might need to respect non-sourced data on the entity in the future
		for (int i=0; i<propertyLists.size(); i++) {
			this.mergeAutofillProperties(target, propertyLists.get(i).getProperties(), pathPrefix + "[" + i + "]", propertyFilter, vocabularyName);
		}
	}
}