package de.uniba.minf.registry.repository;

import java.util.Collection;
import java.util.List;

import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.mongodb.core.MongoOperations;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.index.TextIndexDefinition;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.TextCriteria;

import de.uniba.minf.registry.model.entity.Entity;
import de.uniba.minf.registry.model.vocabulary.VocabularyEntry;

public class VocabularyEntryRepositoryCustomImpl implements VocabularyEntryRepositoryCustom, InitializingBean {
	@Autowired private MongoTemplate mongoTemplate;
	@Autowired private MongoOperations mongoOperations;
	
	@Override
	public void afterPropertiesSet() throws Exception {
		mongoOperations.indexOps(VocabularyEntry.class).ensureIndex(TextIndexDefinition.forAllFields());
	}
	
	@Override
	public long countByDefinition(String definition) {
		return this.findByDefinition(definition).size();
	}

	@Override
	public List<VocabularyEntry> findByDefinition(String definition) {
		final Query query = new Query();
		query.addCriteria(Criteria.where("definitionName").is(definition));
		return mongoTemplate.find(query, VocabularyEntry.class);
	}
	
	@Override
	public List<VocabularyEntry> findByDefinitionAndQuery(String name, String query) {
		Query q = Query.query(Criteria.where("definitionName").is(name))  
				.addCriteria(TextCriteria.forLanguage("en").matching(query));
		
		return mongoTemplate.find(q, VocabularyEntry.class);
	}

	@Override
	public void saveOrUpdate(VocabularyEntry e) {
		Query q = new Query();
		q.addCriteria(Criteria.where("key").is(e.getKey()).and("definitionName").is(e.getDefinitionName()));
		
		VocabularyEntry eExist = mongoTemplate.findOne(q, VocabularyEntry.class);
		if (eExist!=null) {
			e.setUniqueId(eExist.getUniqueId());			
		}
		mongoTemplate.save(e);
	}

	@Override
	public void saveOrUpdateAll(List<VocabularyEntry> vocabularyEntries) {
		if (vocabularyEntries!=null) {
			for (VocabularyEntry ve : vocabularyEntries) {
				this.saveOrUpdate(ve);
			}
		}
	}

	@Override
	public void deleteByVocabularyName(String name) {
		Query q = new Query();
		q.addCriteria(Criteria.where("definitionName").is(name));
		
		mongoTemplate.remove(q, VocabularyEntry.class);
	}
}