package de.uniba.minf.registry.repository;

import org.springframework.data.mongodb.core.query.Update;

import java.util.Collection;
import java.util.Optional;

import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Sort;
import org.springframework.data.mongodb.core.MongoOperations;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.index.TextIndexDefinition;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.CriteriaDefinition;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.TextCriteria;

import de.uniba.minf.registry.model.entity.Entity;

public class EntityRepositoryCustomImpl implements EntityRepositoryCustom, InitializingBean {

	@Autowired private MongoTemplate mongoTemplate;
	@Autowired private MongoOperations mongoOperations;
	
	@Override
	public void afterPropertiesSet() throws Exception {
		mongoOperations.indexOps(Entity.class).ensureIndex(TextIndexDefinition.forAllFields());
	}
	
	@Override
	public Optional<Entity> findLatestByEntityId(String entityId) {
		return findEarliestOrLatestByEntityId(entityId, Sort.Direction.DESC);
	}
	
	@Override
	public Optional<Entity> findEarliestByEntityId(String entityId) {
		return findEarliestOrLatestByEntityId(entityId, Sort.Direction.ASC);
	}
	
	@Override
	public Collection<Entity> findAllLatest() {
		return findLatestByCriteria(Criteria.where("template").is(false));
	}
	
	@Override
	public Collection<Entity> findLatestByDefinition(String definition) {
		return findLatestByCriteria(Criteria.where("definitionName").is(definition).and("template").is(false));
	}
	
	@Override
	public Collection<Entity> findLatestByCriteria(CriteriaDefinition criteria) {
		final Query query = new Query();
		if (criteria!=null) {
			query.addCriteria(criteria);
		}
		query.addCriteria(Criteria.where("deleted").is(false));
		query.addCriteria(Criteria.where("nextVersionUniqueId").isNull());
		return mongoTemplate.find(query, Entity.class);
	}
	
	@Override
	public long countLatestByCriteria(Criteria criteria) {
		final Query query = new Query();
		if (criteria!=null) {
			query.addCriteria(criteria);
		}
		query.addCriteria(Criteria.where("nextVersionUniqueId").isNull().and("deleted").is(false));
		return mongoTemplate.count(query, Entity.class);
	}
	
	@Override
	public Collection<Entity> findLatestByDefinitionAndQuery(String name, String query) {
		Query q = Query.query(Criteria.where("definitionName").is(name).and("template").is(false).and("nextVersionUniqueId").isNull().and("deleted").is(false))  
				.addCriteria(TextCriteria.forLanguage("en").matching(query));
		
		return mongoTemplate.find(q, Entity.class);
	}
	
	private Optional<Entity> findEarliestOrLatestByEntityId(String entityId, Sort.Direction direction) {
		final Query query = new Query();
		query.addCriteria(Criteria.where("entityId").is(entityId).and("deleted").is(false));
		query.with(Sort.by(direction, "creationInstant"));
		
		return Optional.ofNullable(mongoTemplate.findOne(query, Entity.class));	
	}

	@Override
	public long countVersionsByEntityId(String entityId) {
		final Query query = new Query();
		query.addCriteria(Criteria.where("entityId").is(entityId).and("deleted").is(false));
		return mongoTemplate.count(query, Entity.class);
	}

	@Override
	public long countByDefinition(String definition) {
		return this.findLatestByDefinition(definition).size();
	}
	
	@Override
	public long countByImportId(String importId) {
		final Query query = new Query();
		query.addCriteria(Criteria.where("importId").is(importId).and("deleted").is(false));		
		return mongoTemplate.count(query, Entity.class);
	}

	@Override
	public String findDefinitionNameById(String uniqueId) {
		Query q = Query.query(Criteria.where("_id").is(uniqueId));
		q.fields().include("definitionName");
		
		Entity e = mongoTemplate.findOne(q, Entity.class);
		
		if (e!=null) {
			return e.getDefinitionName();
		}
		return null;
	}
	
	@Override
	public boolean getIsLatest(String uniqueId) {
		Query q = Query.query(Criteria.where("_id").is(uniqueId));
		q.fields().include("nextVersionUniqueId");
		
		Entity e = mongoTemplate.findOne(q, Entity.class);
		
		if (e==null || e.getNextVersionUniqueId()!=null) {
			return false;
		}
		return true;
	}

	@Override
	public Collection<Entity> findByImportId(String importId) {
		final Query query = new Query();
		query.addCriteria(Criteria.where("importId").is(importId));		
		return mongoTemplate.find(query, Entity.class);
	}
	
	@Override
	public long removeImportIdByImportId(String importId) {
		final Query query = new Query();
		query.addCriteria(Criteria.where("importId").is(importId));	
		return mongoTemplate.updateMulti(query, new Update().unset("importId"), Entity.class).getModifiedCount();
	}
}
