package de.uniba.minf.registry.migration;

import static de.uniba.minf.registry.model.serialization.base.BaseDefinitionSerializationConstants.*;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;

import org.apache.tika.Tika;
import org.apache.tika.io.TikaInputStream;
import org.apache.tika.metadata.TikaCoreProperties;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.Resource;
import org.springframework.core.io.ResourceLoader;
import org.springframework.core.io.support.ResourcePatternUtils;
import org.springframework.stereotype.Component;
import org.apache.tika.metadata.Metadata;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;

import de.uniba.minf.registry.model.definition.EntityDefinition;
import de.uniba.minf.registry.model.entity.Entity;
import de.uniba.minf.registry.model.vocabulary.VocabularyDefinition;
import de.uniba.minf.registry.model.vocabulary.VocabularyEntry;
import de.unibamberg.minf.core.web.exception.ApplicationSetupException;
import lombok.extern.slf4j.Slf4j;

@Component
@Slf4j
public class InitializationService implements InitializingBean {

	@Value("${paths.definitions:null}")
	private String customDefinitions;
	
	@Value("${paths.entries:null}")
	private String customEntries;
	
	@Autowired ResourceLoader resourceLoader;
	@Autowired @Qualifier("yamlMapper") private ObjectMapper yamlMapper;
	@Autowired @Qualifier("jsonMapper") private ObjectMapper jsonMapper;

	@Autowired private Tika tika;
	
	private List<EntityDefinition> initEntityDefinitions = new ArrayList<>();
	private List<VocabularyDefinition> initVocabularyDefinitions = new ArrayList<>();
	
	@Override
	public void afterPropertiesSet() throws Exception {
		File f;
		if (customDefinitions!=null) {
			f = new File(customDefinitions);
			if (f.exists()) {
				this.scanAndHandleFiles(f, true);
			}
		}
		if (customEntries!=null) {
			f = new File(customEntries);
			if (f.exists()) {
				this.scanAndHandleFiles(f, false);
			}
		}
		
		Resource[] resources = ResourcePatternUtils.getResourcePatternResolver(resourceLoader).getResources("classpath:system_definitions/*");
		for (Resource resource : resources) {
			this.handle(resource.getInputStream(), resource.getFilename(), true);
		}
		
		resources = ResourcePatternUtils.getResourcePatternResolver(resourceLoader).getResources("classpath:system_entries/*");
		for (Resource resource : resources) {
			this.handle(resource.getInputStream(), resource.getFilename(), false);
		}

	}

	private void scanAndHandleFiles(File f, boolean definitions) throws IOException, ApplicationSetupException {
		if (f.isDirectory()) {
			for (File file : f.listFiles()) {
				this.handle(new FileInputStream(file), file.getName(), definitions);
			}
		} else {
			this.handle(new FileInputStream(f), f.getName(), definitions);
		}
	}
		
	private void handle(InputStream is, String fileName, boolean definitions) throws IOException, ApplicationSetupException {
	    TikaInputStream tikaIS = TikaInputStream.get(is);
	    final Metadata metadata = new Metadata();
	    metadata.set(TikaCoreProperties.RESOURCE_NAME_KEY, fileName);
	    String mimeType = tika.detect(tikaIS, metadata);
	    
	    log.info("Read import resource from '{}' as '{}'", fileName, mimeType);
	    
		ObjectMapper objMapper;
		if (mimeType!=null && mimeType.endsWith("/x-yaml") || mimeType.endsWith("/yaml")) {
			objMapper = yamlMapper;
		} else if (mimeType!=null && mimeType.endsWith("/json")) {
			objMapper = jsonMapper;
		} else {
			throw new ApplicationSetupException("Failed to initialize system definitions and entries: unsupported MIME type: " + (mimeType==null ? "NULL" : mimeType), this.getClass());
		}
	
		JsonNode n = objMapper.readTree(is);
		if (n.isArray()) {
			for (JsonNode entry : n) {
				this.handleNode(objMapper, entry, definitions);
			}
		} else {
			this.handleNode(objMapper, n, definitions);
		}
		
		/*
		 * TODO Finalize here:
		 *   * check if n is array and iterate nodes if so
		 *   * check if node has entity or vocabulary fields
		 *   * and treeToValue according to expected type
		 *   * typereference is not required as arrays are resolved
		 */
	}
	
	private void handleNode(ObjectMapper mapper, JsonNode n, boolean definitions) throws JsonProcessingException, IllegalArgumentException {
		if (definitions && n.has(ENTITY_NAME_FIELD)) {
			
			EntityDefinition ed = mapper.treeToValue(n, EntityDefinition.class);
			
		} else if (definitions && n.has(VOCABULARY_FIELD)) {
			
			VocabularyDefinition vd = mapper.treeToValue(n, VocabularyDefinition.class);
			
		} else if (!definitions && n.has("_" + VOCABULARY_FIELD)) {
			
			VocabularyEntry ve = mapper.treeToValue(n, VocabularyEntry.class);
			
		} else {
			
			Entity e = mapper.treeToValue(n, Entity.class);
			
		}
	}
}
