package de.uniba.minf.registry.importer;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.time.Instant;
import java.util.Collection;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import de.uniba.minf.registry.model.Import;
import de.uniba.minf.registry.model.definition.EntityDefinition;
import de.uniba.minf.registry.model.entity.Entity;
import de.uniba.minf.registry.repository.EntityRepository;
import de.uniba.minf.registry.repository.ImportRepository;
import de.uniba.minf.registry.service.EntityService;
import eu.dariah.de.dariahsp.spring.mvc.AuthInfoHelper;
import lombok.extern.slf4j.Slf4j;

@Slf4j
@Component
public class ImportServiceImpl implements ImportService, InitializingBean, PropertyChangeListener {

	@Autowired private EntityService entityService;
	@Autowired private EntityRepository entityRepo;
	@Autowired private ImportRepository importRepo;
	
	@Autowired private AuthInfoHelper authinfoHelper;
	
	protected ExecutorService importExecutor;
	private int maxPoolSize = 5;
	
	@Override
	public void afterPropertiesSet() throws Exception {
		importExecutor = Executors.newFixedThreadPool(maxPoolSize);
	}
	
	@Override
	public long deleteImport(String uniqueId) {	
		Optional<Import> i = importRepo.findById(uniqueId);
		if (i.isEmpty()) {
			return 0;
		}
		return this.deleteImport(i.get());
	} 
	
	@Override
	public long deleteImport(Import i) {	
		long count;
		if (i.isReadOnly()) {
			Collection<Entity> entities = entityRepo.findByImportId(i.getUniqueId());
			count = entities.size();
			entityRepo.deleteAll(entities);
			log.info("Deleting read-only import, deleted {} associated entites", count);
		} else {
			count = entityRepo.removeImportIdByImportId(i.getUniqueId());
			log.info("Deleting modifiable import, unassociated {} entites", count);
		}
		
		importRepo.delete(i);
		log.info("Deleted import: {}", i.getUniqueId());
		
		return count;
	}

	@Override
	public Import importEntities(List<Entity> entities, List<Boolean> isDraft, EntityDefinition ed, boolean readonly, String templateId) {
		String userId = authinfoHelper.getAuth().getUserId();
		
		Import im = new Import();
		im.setReadOnly(readonly);
		im.setTemplateId(templateId);
		im.setUserUniqueId(userId);
		im.setCreationInstant(Instant.now());
		
		importRepo.save(im);
		
		for (int i=0; i<entities.size(); i++) {

			entities.get(i).setUserUniqueId(userId);
			entities.get(i).setImportId(im.getUniqueId());
			entities.get(i).setReadOnly(readonly);
			entities.get(i).setDraft(isDraft.get(i));
			
			// Entities that are related by a vocabulary need to be persisted first 
			entityService.setOrCreateRelatedEntities(entities.get(i), ed);
			
			// Save entity
			entityService.save(entities.get(i));
		}
		
		return im;
	}

	@Override
	public void importAsync(ImportRunner ir) {
		
		ir.addPropertyChangeListener(this);
		
		this.importExecutor.execute(ir);
	}

	@Override
	public void propertyChange(PropertyChangeEvent arg0) {
		log.debug(arg0.getPropertyName());
	}

	
}
