package de.uniba.minf.registry.controller.user;

import java.time.Instant;
import java.util.List;
import java.util.Locale;
import java.util.UUID;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import de.uniba.minf.core.rest.exception.ApiInsufficientPermissionsException;
import de.uniba.minf.core.rest.exception.ApiItemNotFoundException;
import de.uniba.minf.core.rest.model.RestItemResponse;
import de.uniba.minf.core.rest.model.RestResponse.ApiActions;
import de.uniba.minf.core.rest.model.RestItemsResponse;
import de.uniba.minf.registry.model.PersistedUser;
import de.uniba.minf.registry.model.RegistryAccessToken;
import de.uniba.minf.registry.repository.AccessTokenRepository;
import eu.dariah.de.dariahsp.spring.mvc.AuthInfoHelper;
import eu.dariah.de.dariahsp.spring.mvc.model.AuthPojo;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.servlet.http.HttpServletRequest;
import lombok.extern.slf4j.Slf4j;

@Slf4j
@Tag(name = "Access Token", description = "API methods for user access tokens")
@RestController
@RequestMapping("/api/v1/tokens/{userUID}")
public class AccessTokenController extends BaseUserRestController<RegistryAccessToken> {
	
	public AccessTokenController() {
		super("/api/v1/tokens");
	}

	private static final String ITEM_TYPE = "accessToken";
	
	@Autowired private AccessTokenRepository accessTokenRepo;
	
	@GetMapping
	public RestItemsResponse getUserAccessTokens(@PathVariable("userUID") String userUID, HttpServletRequest request, Locale locale) throws ApiInsufficientPermissionsException, ApiItemNotFoundException {
		AuthPojo authPojo = authInfoHelper.getAuth();
		PersistedUser user = this.checkCanAccessUser(authPojo, userUID);
		
		RestItemsResponse resp = new RestItemsResponse();
		
		List<RegistryAccessToken> tokens = accessTokenRepo.findByUserId(user.getUniqueId());
		
		// Hide API key
		tokens.stream().forEach(t -> t.setUniqueId(null));
		
		resp.setSize(tokens.size());
		resp.setItems(this.getItems(tokens, request.getRequestURL().toString()));
		resp.setLinks(this.getLinks(request.getRequestURL().toString()));
		return resp;
	}
	
	@GetMapping("/{tokenUID}")
	public RestItemResponse getUserAccessToken(@PathVariable("userUID") String userUID, @PathVariable("tokenUID") String tokenUID, HttpServletRequest request, Locale locale) throws ApiInsufficientPermissionsException, ApiItemNotFoundException {
		RegistryAccessToken t = this.getToken(userUID, tokenUID);
		// Hide API key
		t.setUniqueId(null);
		return this.getItemResponse(t, request, ApiActions.GET);
	}
	
	@DeleteMapping("/{tokenUID}")
	public RestItemResponse deleteUserAccessToken(@PathVariable("userUID") String userUID, @PathVariable("tokenUID") String tokenUID, HttpServletRequest request, Locale locale) throws ApiInsufficientPermissionsException, ApiItemNotFoundException {
		RegistryAccessToken token = this.getToken(userUID, tokenUID);
		accessTokenRepo.deleteById(token.getId());	
		
		// Hide API key
		token.setUniqueId(null);
		return this.getItemResponse(token, request, ApiActions.DELETED);
	}
	
	@PostMapping("/{tokenUID}")
	public RestItemResponse updateUserAccessToken(@PathVariable("userUID") String userUID, @PathVariable("tokenUID") String tokenUID, @RequestBody RegistryAccessToken updatedToken, HttpServletRequest request, Locale locale) throws ApiInsufficientPermissionsException, ApiItemNotFoundException {
		RegistryAccessToken token = this.getToken(userUID, tokenUID);

		this.fillToken(token, updatedToken);
		accessTokenRepo.save(token);
		
		// Hide API key
		token.setUniqueId(null);
		
		return this.getItemResponse(token, request, ApiActions.UPDATED);
	}
		
	@PostMapping
	public RestItemResponse saveNewUserAccessToken(@PathVariable("userUID") String userUID, @RequestBody(required=false) RegistryAccessToken newToken, HttpServletRequest request, Locale locale) throws ApiInsufficientPermissionsException, ApiItemNotFoundException {
		AuthPojo authPojo = authInfoHelper.getAuth();
		PersistedUser user = this.checkCanAccessUser(authPojo, userUID);
		
		RegistryAccessToken token = new RegistryAccessToken();
		this.fillToken(token, newToken);
		token.setUserId(user.getUniqueId());
		token.setUniqueId(UUID.randomUUID().toString());
		token.setCreationInstant(Instant.now());
		
		accessTokenRepo.save(token);
		
		// Not hidden after inital save
		
		return this.getItemResponse(token, request, ApiActions.CREATED);
	}
			
	private RegistryAccessToken getToken(String userUID, String tokenUID) throws ApiItemNotFoundException, ApiInsufficientPermissionsException {
		AuthPojo authPojo = authInfoHelper.getAuth();
		PersistedUser user = this.checkCanAccessUser(authPojo, userUID);
		
		RegistryAccessToken token = accessTokenRepo.findByIdAndUserId(tokenUID, user.getUniqueId());
		if (token==null) {
			throw new ApiItemNotFoundException(ITEM_TYPE, tokenUID);			
		}
		return token;
	}
	
	private void fillToken(RegistryAccessToken saveToken, RegistryAccessToken updateToken) {
		if (updateToken!=null) {
			saveToken.setAllowedAdresses(updateToken.getAllowedAdresses());
			saveToken.setExpires(updateToken.getExpires());
			saveToken.setName(updateToken.getName());
		}
		// Currently no distinction between write and read tokens
		saveToken.setWriteAccess(true);
	}
}
