package de.uniba.minf.registry.controller;

import java.util.Collection;
import java.util.List;
import java.util.Locale;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonMappingException;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;

import de.uniba.minf.core.rest.controller.BaseRestController;
import de.uniba.minf.core.rest.model.RestItemsResponse;
import de.uniba.minf.registry.model.vocabulary.VocabularyDefinition;
import de.uniba.minf.registry.pojo.VocabularyDefinitionPojo;
import de.uniba.minf.registry.pojo.converter.VocabularyDefinitionConverter;
import de.uniba.minf.registry.repository.VocabularyDefinitionRepository;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.servlet.http.HttpServletRequest;
import lombok.extern.slf4j.Slf4j;

@Slf4j
@Tag(name = "Vocabulary definitions", description = "Datamodel specifications of for vocabulary entities")
@RestController
@RequestMapping("/api/v1/def/vocabularies")
public class VocabularyDefinitionController extends BaseRestController<VocabularyDefinitionPojo> {

	public VocabularyDefinitionController() {
		super("/api/v1/def/vocabularies");
	}

	@Autowired @Qualifier("yamlMapper") private ObjectMapper yamlMapper;
	@Autowired @Qualifier("jsonMapper") private ObjectMapper jsonMapper;
	
	@Autowired private VocabularyDefinitionRepository vocabularyDefinitionRepo;
	@Autowired private VocabularyDefinitionConverter vocabularyDefinitionConverter;
	
	@PostMapping(consumes = {"application/x-yaml", "text/yaml"})
	public void postYaml(@RequestBody String updatedApi, Locale locale) throws JsonMappingException, JsonProcessingException {
		
		VocabularyDefinition vd = yamlMapper.readValue(updatedApi, VocabularyDefinition.class);
		
		vocabularyDefinitionRepo.save(vd);
		
		log.debug("" + vd.getVersion());
	}
	
	@PostMapping(consumes = {"application/json"})
	public void postJson(@RequestBody String updatedApi, Locale locale) throws JsonMappingException, JsonProcessingException {
		JsonNode n = jsonMapper.readTree(updatedApi);
		if (n.isArray()) {
			for (JsonNode n1 : n) {
				vocabularyDefinitionRepo.save(jsonMapper.treeToValue(n1, VocabularyDefinition.class));
			}
		} else {
			vocabularyDefinitionRepo.save(jsonMapper.treeToValue(n, VocabularyDefinition.class));
		}
	}
	
	
	@GetMapping
	public RestItemsResponse get(HttpServletRequest request, Locale locale) {
		RestItemsResponse response = new RestItemsResponse();
		
		Collection<VocabularyDefinitionPojo> vocabularyDefinitions = vocabularyDefinitionConverter.convertPojos(vocabularyDefinitionRepo.findAllLatest());
		
		response.setSize(vocabularyDefinitions.size());
		response.setLinks(this.getLinks(request.getRequestURL().toString()));
		response.setItems(this.getItems(vocabularyDefinitions, request.getRequestURL().toString()));
		
		return response;
	}
	
	@GetMapping("{id}")
	public VocabularyDefinitionPojo get(@PathVariable("id") String id, Locale locale) {
		return vocabularyDefinitionConverter.convertPojo(vocabularyDefinitionRepo.findById(id).get());
	}
	
	@GetMapping("/by-name/{name}")
	public VocabularyDefinitionPojo getByName(@PathVariable("name") String name, Locale locale) {
		return vocabularyDefinitionConverter.convertPojo(vocabularyDefinitionRepo.findCurrentByName(name));
	}
}
