package de.uniba.minf.registry.controller;

import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;
import java.time.temporal.TemporalAccessor;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Locale;
import java.util.Optional;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import de.uniba.minf.core.rest.controller.BaseRestController;
import de.uniba.minf.core.rest.exception.ApiInsufficientPermissionsException;
import de.uniba.minf.core.rest.exception.ApiItemNotFoundException;
import de.uniba.minf.core.rest.model.RestItemsResponse;
import de.uniba.minf.core.rest.model.RestResponse.ApiActions;
import de.uniba.minf.registry.model.PersistedUser;
import de.uniba.minf.registry.model.UserGroup;
import de.uniba.minf.registry.model.UserGroupAssignment;
import de.uniba.minf.registry.pojo.UserGroupAssignmentPojo;
import de.uniba.minf.registry.pojo.converter.UserGroupAssignmentConverter;
import de.uniba.minf.registry.repository.UserGroupRepository;
import de.uniba.minf.registry.repository.UserRepository;
import eu.dariah.de.dariahsp.spring.mvc.model.AuthPojo;
import jakarta.servlet.http.HttpServletRequest;
import lombok.extern.slf4j.Slf4j;

@Slf4j
@RestController
@RequestMapping("/api/v1/uga")
public class UserGroupAssignmentController extends BaseRestController<UserGroupAssignmentPojo> {
	
	@Autowired private UserGroupRepository userGroupRepository;
	@Autowired private UserRepository userRepository;
	@Autowired private UserGroupAssignmentConverter userGroupAssignmentConverter;
	
	public UserGroupAssignmentController() {
		super("/api/v1/uga");
	}

	@GetMapping
	public RestItemsResponse getUserGroupAssignments(HttpServletRequest request, Locale locale) throws ApiInsufficientPermissionsException, ApiItemNotFoundException {
		AuthPojo authPojo = authInfoHelper.getAuth();
		
		RestItemsResponse resp = new RestItemsResponse();
		
		List<UserGroup> assignedGroups = userGroupRepository.findByUserUniqueId(authPojo.getUserId());
		List<UserGroupAssignmentPojo> ugas = new ArrayList<>();
		List<UserGroupAssignmentPojo> ugaPart;
		for (UserGroup g : assignedGroups) {
			boolean userCanWrite = g.getOwnerUniqueId().equals(authPojo.getUserId()) || 
					g.getUserGroupAssignments().stream().anyMatch(ga -> ga.getUserUniqueId()!=null && ga.getUserUniqueId().equals(authPojo.getUserId()) && ga.isWriteAccess());
			
			ugaPart = userGroupAssignmentConverter.convertPojo(g);
			if (userCanWrite) {
				ugaPart.forEach(uga -> uga.setCurrentUserCanEdit(true));
			}
			ugas.addAll(ugaPart);
		}
		
		resp.setSize(ugas.size());
		resp.setItems(this.getItems(ugas, request.getRequestURL().toString()));
		resp.setLinks(this.getLinks(request.getRequestURL().toString()));
		return resp;
	}
	
	@PostMapping({"/{uniqueId}", ""})
	public RestItemsResponse saveNewUserGroup(@PathVariable(required=false,value="uniqueId") String uniqueId, @RequestBody(required=false) UserGroupAssignmentPojo newGroup, HttpServletRequest request, Locale locale) throws ApiInsufficientPermissionsException, ApiItemNotFoundException {
		AuthPojo authPojo = authInfoHelper.getAuth();

		UserGroup ug;
		RestItemsResponse resp;
		if (uniqueId!=null) {
			Optional<UserGroup> optUg = userGroupRepository.findById(uniqueId);
			if (!optUg.isPresent()) {
				throw new ApiItemNotFoundException("userGroup", uniqueId);
			}
			ug = optUg.get();
			resp = this.getUserGroupResponse(ug, request, ApiActions.UPDATED);
		} else {
			ug = new UserGroup();
			ug.setOwnerUniqueId(authPojo.getUserId());	
			resp = this.getUserGroupResponse(ug, request, ApiActions.CREATED);
		}		
		ug.setName(newGroup.getName());		
		userGroupRepository.save(ug);
				
		return resp;
	}
		
	@PostMapping("/{uniqueId}/user")
	public RestItemsResponse saveUserInGroup(@PathVariable("uniqueId") String uniqueId, @RequestBody UserGroupAssignmentPojo newUga, HttpServletRequest request, Locale locale) throws ApiInsufficientPermissionsException, ApiItemNotFoundException {
		Optional<UserGroup> optUg = userGroupRepository.findById(uniqueId);
		if (optUg.isEmpty()) {
			throw new ApiItemNotFoundException("userGroup", uniqueId);
		}
		UserGroup ug = optUg.get();
		if (ug.getUserGroupAssignments()==null) {
			ug.setUserGroupAssignments(new ArrayList<>());
		}
		
		Optional<PersistedUser> user = userRepository.findByUsername(newUga.getUsername());
		Optional<UserGroupAssignment> existUga;
		UserGroupAssignment uga;
		// Already owner
		if (user.isPresent() && user.get().getUniqueId().equals(ug.getOwnerUniqueId())) {
			return this.getUserGroupResponse(ug, request, ApiActions.UNCHANGED);
		}
		
		if (newUga.getAssignmentUniqueId()!=null) { 
			existUga = ug.getUserGroupAssignments().stream().filter(u -> u.getUniqueId().equals(newUga.getAssignmentUniqueId())).findFirst();
		} else if (user.isPresent()) {
			existUga = ug.getUserGroupAssignments().stream().filter(u -> u.getUserUniqueId()!=null && u.getUserUniqueId().equals(user.get().getUniqueId())).findFirst();
		} else {
			existUga = ug.getUserGroupAssignments().stream().filter(u -> u.getUsername().equals(newUga.getUsername())).findFirst();
		}
		
		
		if (existUga.isPresent()) {
			uga = existUga.get();
		} else {
			uga = new UserGroupAssignment();
			ug.getUserGroupAssignments().add(uga);
		}
		uga.setUsername(newUga.getUsername());
		uga.setUserUniqueId(user.isPresent() ? user.get().getUniqueId() : null);
		uga.setWriteAccess(newUga.isWriteAccess());
		
		if (newUga.getExpirationString()!=null && !newUga.getExpirationString().isEmpty()) {
			try {
				LocalDate localDate = LocalDate.parse(newUga.getExpirationString(), DateTimeFormatter.ISO_DATE); 
				LocalDateTime localDateTime = localDate.atTime(0, 0);
				ZonedDateTime zonedDateTime = localDateTime.atZone(ZoneId.of("UTC"));
				uga.setExpiration(zonedDateTime.toInstant());
			} catch (DateTimeParseException ex) {
				log.warn("Failed to parse expiration string" , ex);
			}
		} else {
			uga.setExpiration(null);
		}
		userGroupRepository.save(ug);
		
		return this.getUserGroupResponse(ug, request, ApiActions.UPDATED);
	}
	
	@DeleteMapping("/{uniqueId}/user/{assignmentUniqueId}")
	public RestItemsResponse removeUserFromGroup(@PathVariable("uniqueId") String uniqueId, @PathVariable("assignmentUniqueId") String assignmentUniqueId, HttpServletRequest request, Locale locale) throws ApiInsufficientPermissionsException, ApiItemNotFoundException {
		Optional<UserGroup> optUg = userGroupRepository.findById(uniqueId);
		if (optUg.isEmpty()) {
			throw new ApiItemNotFoundException("userGroup", uniqueId);
		}
		UserGroup ug = optUg.get();
		if (ug.getUserGroupAssignments()!=null) {
			List<UserGroupAssignment> removeUga = ug.getUserGroupAssignments().stream().filter(uga -> uga.getUniqueId().equals(assignmentUniqueId)).toList();
			if (!removeUga.isEmpty()) {
				ug.getUserGroupAssignments().removeAll(removeUga);
				userGroupRepository.save(ug);
				return this.getUserGroupResponse(ug, request, ApiActions.UPDATED);
			}
		}
		return this.getUserGroupResponse(ug, request, ApiActions.UNCHANGED);
	}
	
	@DeleteMapping("/{uniqueId}")
	public RestItemsResponse deleteUserGroup(@PathVariable("uniqueId") String uniqueId, HttpServletRequest request, Locale locale) throws ApiInsufficientPermissionsException, ApiItemNotFoundException {
		Optional<UserGroup> optUg = userGroupRepository.findById(uniqueId);
		if (optUg.isEmpty()) {
			throw new ApiItemNotFoundException("userGroup", uniqueId);
		}
		userGroupRepository.delete(optUg.get());
		
		return this.getUserGroupResponse(optUg.get(), request, ApiActions.DELETED);
	}
	
	private RestItemsResponse getUserGroupResponse(UserGroup ug, HttpServletRequest request, ApiActions apiAction) {
		RestItemsResponse resp = new RestItemsResponse();	
		List<UserGroupAssignmentPojo> uga = userGroupAssignmentConverter.convertPojo(ug);
		
		resp.setAction(apiAction);
		resp.setSize(uga.size());
		resp.setItems(this.getItems(uga, request.getRequestURL().toString()));
		resp.setLinks(this.getLinks(request.getRequestURL().toString()));
		return resp;
	}
}