package de.uniba.minf.registry.model.validation;

import static de.uniba.minf.registry.model.validation.ValidationConstants.*;

import java.util.function.UnaryOperator;

import am.ik.yavi.builder.ValidatorBuilder;
import am.ik.yavi.constraint.BooleanConstraint;
import am.ik.yavi.constraint.CharSequenceConstraint;
import am.ik.yavi.constraint.DoubleConstraint;
import am.ik.yavi.constraint.IntegerConstraint;
import am.ik.yavi.core.Validator;
import de.uniba.minf.registry.model.BasePropertyValue;
import de.uniba.minf.registry.model.definition.PropertyDefinition;
import de.uniba.minf.registry.model.definition.SimplePropertyDefinition;
import de.uniba.minf.registry.model.definition.VocabularyPropertyDefinition;
import de.uniba.minf.registry.model.definition.SimplePropertyDefinition.SIMPLE_TYPES;
import de.uniba.minf.registry.model.entity.AutoqueryEntityLookupService;
import de.uniba.minf.registry.model.validation.exception.ValidationConfigurationException;
import de.uniba.minf.registry.model.vocabulary.ValidationEntityService;
import de.uniba.minf.registry.model.vocabulary.VocabularyLookupService;
import lombok.Getter;

public class PropertyValueValidator<T extends BasePropertyValue<?>> extends ValidatorBuilder<T>  {
	@Getter private final Class<T> validatedClass;	
	private Validator<T> validator = null;
	
	private static final ValidateUrlConstraint urlC = new ValidateUrlConstraint();
	
	@Getter private final VocabularyLookupService vocabularyLookupService;
	@Getter private final ValidationEntityService entityService;
	@Getter private final AutoqueryEntityLookupService autoqueryLookupService;
	
	public PropertyValueValidator(Class<T> validatedClass, PropertyDefinition pd) throws ValidationConfigurationException {
		this(validatedClass, pd, null, null, null);
	}
	
	public PropertyValueValidator(Class<T> validatedClass, PropertyDefinition pd, VocabularyLookupService vocabularyLookupService, AutoqueryEntityLookupService autoqueryLookupService, ValidationEntityService entityService) throws ValidationConfigurationException {
		this.validatedClass = validatedClass;
		this.vocabularyLookupService = vocabularyLookupService;
		this.entityService = entityService;
		this.autoqueryLookupService = autoqueryLookupService;
		this.buildValidation(pd);
	}
	
	public Validator<T> getValidator() {
		if (this.validator==null) {
			this.validator = this.build();
		} 
		return this.validator;
	}
		
	private void buildValidation(PropertyDefinition pd) throws ValidationConfigurationException {
		SIMPLE_TYPES validationType = null;
		if (SimplePropertyDefinition.class.isAssignableFrom(pd.getClass())) {
			validationType = SimplePropertyDefinition.class.cast(pd).getType();
		}
		
		if (validationType==null || validationType.equals(SIMPLE_TYPES.TEXT) || validationType.equals(SIMPLE_TYPES.LONGTEXT) || validationType.equals(SIMPLE_TYPES.DATE)) {
			this.buildTextValidation(pd);
			if (VocabularyPropertyDefinition.class.isAssignableFrom(pd.getClass())) {
				VocabularyPropertyDefinition vpd = VocabularyPropertyDefinition.class.cast(pd);
				if (vpd.isQuery() && vpd.isStrict()) {
					QueryVocabularyConstraint qvc = new QueryVocabularyConstraint(vpd.getVocabulary(), getVocabularyLookupService(), getAutoqueryLookupService(), entityService, vpd.isEntity());
					this.applyStringConstraint(x -> x.predicate(qvc));
				}
			}
		} else if (validationType.equals(SIMPLE_TYPES.INT)) {
			this.buildIntegerValidation(pd);
		} else if (validationType.equals(SIMPLE_TYPES.FLOAT)) {
			this.buildDoubleValidation(pd);
		} else if (validationType.equals(SIMPLE_TYPES.BOOLEAN)) {
			this.buildBooleanValidation(pd);
		}
	}
	
	private void buildTextValidation(PropertyDefinition pd) throws ValidationConfigurationException {
		if (pd.getValid()!=null) {
			for (String c : pd.getValid()) {
				NumParamConstraint nC = new NumParamConstraint(c);
				ValidationMethods method = nC.getAndAssertValidationMethod();
				
				//
				// TODO: add contains, pattern but they require changes to the NumParamConstraint (not NumParam but TextParam)
				// 			- https://yavi.ik.am/#containscharsequence
				//			- https://yavi.ik.am/#patternstring
				//
				if (method.equals(ValidationMethods.NOTNULL)) {
					this.applyStringConstraint(x -> x.notNull());
				} else if (method.equals(ValidationMethods.NOTBLANK)) {
					this.applyStringConstraint(x -> x.notBlank());
				} else if (method.equals(ValidationMethods.GREATERTHAN)) {
					this.applyStringConstraint(x -> x.greaterThan(nC.getNum()));
				} else if (method.equals(ValidationMethods.GREATERTHANOREQUAL)) {
					this.applyStringConstraint(x -> x.greaterThanOrEqual(nC.getNum()));
				} else if (method.equals(ValidationMethods.LESSTHAN)) {
					this.applyStringConstraint(x -> x.lessThan(nC.getNum()));
				} else if (method.equals(ValidationMethods.LESSTHANOREQUAL)) {
					this.applyStringConstraint(x -> x.lessThanOrEqual(nC.getNum()));
				} else if (method.equals(ValidationMethods.URL)) {
					// TODO: Maybe split here between validation warnings and errors?
					//this.applyStringConstraint(x -> x.url());
					this.applyStringConstraint(x -> x.predicate(urlC));					
				} else {
					throw new ValidationConfigurationException("Configured constraint method not applicable to type, method: " + method + ", type: " + "text");
				}
			}
		}
	}

	private void buildIntegerValidation(PropertyDefinition pd) throws ValidationConfigurationException {
		if (pd.getValid()!=null) {
			for (String c : pd.getValid()) {
				NumParamConstraint nC = new NumParamConstraint(c);
				ValidationMethods method = nC.getAndAssertValidationMethod();
				
				if (method.equals(ValidationMethods.NOTNULL)) {
					this.applyIntegerConstraint(x -> x.notNull());
				} else if (method.equals(ValidationMethods.GREATERTHAN)) {
					this.applyIntegerConstraint(x -> x.greaterThan(nC.getNum()));
				} else if (method.equals(ValidationMethods.GREATERTHANOREQUAL)) {
					this.applyIntegerConstraint(x -> x.greaterThanOrEqual(nC.getNum()));
				} else if (method.equals(ValidationMethods.LESSTHAN)) {
					this.applyIntegerConstraint(x -> x.lessThan(nC.getNum()));
				} else if (method.equals(ValidationMethods.LESSTHANOREQUAL)) {
					this.applyIntegerConstraint(x -> x.lessThanOrEqual(nC.getNum()));
				} else {
					throw new ValidationConfigurationException("Configured constraint method not applicable to type, method: " + method + ", type: " + "text");
				}	
			}
		}
	}
	
	private void buildDoubleValidation(PropertyDefinition pd) throws ValidationConfigurationException {
		if (pd.getValid()!=null) {
			for (String c : pd.getValid()) {
				NumParamConstraint nC = new NumParamConstraint(c);
				ValidationMethods method = nC.getAndAssertValidationMethod();
				
				if (method.equals(ValidationMethods.NOTNULL)) {
					this.applyDoubleConstraint(x -> x.notNull());
				} else if (method.equals(ValidationMethods.GREATERTHAN)) {
					this.applyDoubleConstraint(x -> x.greaterThan(nC.numAsDouble()));
				} else if (method.equals(ValidationMethods.GREATERTHANOREQUAL)) {
					this.applyDoubleConstraint(x -> x.greaterThanOrEqual(nC.numAsDouble()));
				} else if (method.equals(ValidationMethods.LESSTHAN)) {
					this.applyDoubleConstraint(x -> x.lessThan(nC.numAsDouble()));
				} else if (method.equals(ValidationMethods.LESSTHANOREQUAL)) {
					this.applyDoubleConstraint(x -> x.lessThanOrEqual(nC.numAsDouble()));
				} else {
					throw new ValidationConfigurationException("Configured constraint method not applicable to type, method: " + method + ", type: " + "text");
				}	
			}
		}
	}
	
	private void buildBooleanValidation(PropertyDefinition pd) throws ValidationConfigurationException {
		if (pd.getValid()!=null) {
			for (String c : pd.getValid()) {
				NumParamConstraint nC = new NumParamConstraint(c);
				ValidationMethods method = nC.getAndAssertValidationMethod();
				
				if (method.equals(ValidationMethods.NOTNULL)) {
					this.applyBooleanConstraint(x -> x.notNull());
				} else if (method.equals(ValidationMethods.ISTRUE)) {
					this.applyBooleanConstraint(x -> x.isTrue());
				} else if (method.equals(ValidationMethods.ISFALSE)) {
					this.applyBooleanConstraint(x -> x.isFalse());
				} else {
					throw new ValidationConfigurationException("Configured constraint method not applicable to type, method: " + method + ", type: " + "text");
				}	
			}
		}
	}

	private void applyStringConstraint(UnaryOperator<CharSequenceConstraint<T, String>> c) {
		this._string(x -> x.getValue().toString(), "_" + SIMPLE_TYPES.TEXT.name().toLowerCase(), c);
	}
	
	private void applyIntegerConstraint(UnaryOperator<IntegerConstraint<T>> c) {
		this._integer(x -> (int)x.getValue(), "_" + SIMPLE_TYPES.INT.name().toLowerCase(), c);		
	}
	
	private void applyDoubleConstraint(UnaryOperator<DoubleConstraint<T>> c) {
		this._double(x -> (double)x.getValue(), "_" + SIMPLE_TYPES.FLOAT.name().toLowerCase(), c);
	}
	
	private void applyBooleanConstraint(UnaryOperator<BooleanConstraint<T>> c) {
		this._boolean(x -> (boolean)x.getValue(), "_" + SIMPLE_TYPES.BOOLEAN.name().toLowerCase(), c);
	}
}
