package de.uniba.minf.registry.model.definition;

import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;

import org.springframework.data.annotation.Id;
import org.springframework.data.mongodb.core.index.CompoundIndex;
import org.springframework.data.mongodb.core.index.Indexed;

import de.uniba.minf.core.rest.model.Identifiable;
import de.uniba.minf.registry.model.serialization.base.SerializationMessage;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;

@Slf4j
@Data
@CompoundIndex(def = "{'name': 1, 'version': 1}", unique = true)
public abstract class BaseDefinition implements Identifiable {
	private static final long serialVersionUID = 4908093821116664728L;

	public static enum PROPERTY_FLAGS { AUTOFILL, AUTOQUERY, ENTITY_VOCABULARY, EXTERNAL_IDENTIFIER, RELATION, PRESENTATION, IMAGE_RENDERING }
	
	/**
	 * ID of the version -> changes with every update
	 */
	@Id
	private String uniqueId;
	
	/**
	 * Identifier of the definition
	 */
	@Indexed
	private String name;
	
	/**
	 * Informal use only
	 */
	private long version;
	
	/**
	 * Creation instant of the version
	 */
	private Instant creationInstant;
	private String userUniqueId;	
	
	private String label;
	private String messageCode;
	
	private String endpointUrl;
	private String endpointSearchUrl;
	private String endpointGetUrl;
	private String endpointMethod;
	
	public String getEndpointSearchUrl() {
		if (this.endpointSearchUrl!=null) {
			return this.endpointSearchUrl;
		}
		return this.endpointUrl;
	}
	
	public String getEndpointGetUrl() {
		if (this.endpointGetUrl!=null) {
			return this.endpointGetUrl;
		}
		return this.endpointUrl;
	}
	
	@Indexed
	private String nextVersionUniqueId;
	
	private boolean system;
	
	public boolean isRemote() {
		return this.endpointUrl!=null || this.getEndpointGetUrl()!=null || this.getEndpointSearchUrl()!=null;
	}
	
	public String getMessageCode() {
		return messageCode!=null ? messageCode : ("~" + name);
	}
	
	private List<SerializationMessage> messages;
	private List<PropertyDefinitionBlock> propertyBlocks;
	
	private List<String> previewPropertyIdentifiers;
	
	public List<PropertyDefinition> getPreviewProperties() {
		final List<PropertyDefinition> previewProperties = new ArrayList<>();
		if (previewPropertyIdentifiers!=null && !previewPropertyIdentifiers.isEmpty()) {
			for (String previewPropertyIdentifier : previewPropertyIdentifiers) {
				Optional<PropertyDefinition> prevPd = this.getAllProperties().stream().filter(p -> p.getIdentifier().equals(previewPropertyIdentifier)).findFirst();
				if (prevPd.isPresent()) {
					previewProperties.add(prevPd.get());
				} else {
					log.warn("Skipping unknown preview property identifier: " + previewPropertyIdentifier);
				}
			}
		}
		if (previewProperties.size()<5) {
			previewProperties.addAll(this.getAllProperties().stream().limit(5-previewProperties.size()).toList());
		}
		
		return previewProperties;
	}
	
	public List<PropertyDefinition> getOrderedProperties() {
		final List<PropertyDefinition> orderedProperties = new ArrayList<>();
		orderedProperties.addAll(this.getPreviewProperties());
		orderedProperties.addAll(this.getAllProperties().stream().filter(p -> !orderedProperties.contains(p) && !p.isSystem()).toList());
		return orderedProperties;
	}
	
	public void addMessage(SerializationMessage message) {
		if (messages==null) {
			messages = new ArrayList<>();
		}
		messages.add(message);
	}
	
	public PropertyDefinition getDefinition(String identifier) {
		String[] path = identifier.split("\\.");
		return getDefinition(this.getAllProperties(), path);
	}
	
	public PropertyDefinition getDefinition(String... path) {
		return getDefinition(this.getAllProperties(), path);
	}
	
	public boolean hasAutoqueryProperties() {
		return !this.getAutoqueryProperties().isEmpty();
	}
	
	public boolean hasAutofillProperties() {
		return !this.getAutofillProperties().isEmpty();
	}
	
	public boolean hasImageRenderingProperties() {
		return !this.getImageRenderingProperties().isEmpty();
	}
	
	public List<PropertyDefinition> getAllProperties() {
		return this.propertyBlocks.stream().flatMap(b -> b.getProperties().stream()).toList();
	}
	
	public List<VocabularyPropertyDefinition> getAutoqueryProperties() {
		return getProperties(this.getAllProperties(), PROPERTY_FLAGS.AUTOQUERY);
	}
	
	public List<RelationPropertyDefinition> getRelationProperties() {
		return getProperties(this.getAllProperties(), PROPERTY_FLAGS.RELATION);
	}
	
	public List<VocabularyPropertyDefinition> getAutofillProperties() {
		return getProperties(this.getAllProperties(), PROPERTY_FLAGS.AUTOFILL);
	}
		
	public List<VocabularyPropertyDefinition> getEntityVocabularyProperties() {
		return getProperties(this.getAllProperties(), PROPERTY_FLAGS.ENTITY_VOCABULARY);
	}
	
	public List<PropertyDefinition> getExternalIdentifierProperties() {
		return getProperties(this.getAllProperties(), PROPERTY_FLAGS.EXTERNAL_IDENTIFIER);
	}
	
	public List<PropertyDefinition> getImageRenderingProperties() {
		return getProperties(this.getAllProperties(), PROPERTY_FLAGS.IMAGE_RENDERING);
	}
	
	public static <T extends PropertyDefinition> List<T> getProperties(List<PropertyDefinition> properties, PROPERTY_FLAGS flag) {
		List<T> matchingDefinitions = new ArrayList<>();
		if (properties!=null) {
			for (PropertyDefinition pd : properties) {
				// Special treatment for relation properties as they are hierarchical,
				//  but if requesting PROPERTY_FLAGS.RELATION no recursion is indicated
				if (flag.equals(PROPERTY_FLAGS.RELATION)) {
					if (pd.isRelation()) {
						matchingDefinitions.add((T)pd);
					}
				} else {
					if (getPropertyIsMatch(pd, flag)) {
						matchingDefinitions.add((T)pd);
					}
				}
				
				// Recursion
				if (pd.isHierarchical()) {
					matchingDefinitions.addAll(getProperties(HierarchicalPropertyDefinition.class.cast(pd).getProperties(), flag));
				}
			}
		}
		return matchingDefinitions;
	}
	
	private static boolean getPropertyIsMatch(PropertyDefinition pd, PROPERTY_FLAGS flag) {
		
		if ( (flag.equals(PROPERTY_FLAGS.AUTOQUERY) && (!pd.isVocabulary() || !VocabularyPropertyDefinition.class.cast(pd).isAutoQuery())) ||
			 (flag.equals(PROPERTY_FLAGS.AUTOFILL) && (!pd.isVocabulary() || !VocabularyPropertyDefinition.class.cast(pd).isAutofill())) ||	
			 (flag.equals(PROPERTY_FLAGS.ENTITY_VOCABULARY) && (!pd.isVocabulary() || !VocabularyPropertyDefinition.class.cast(pd).isEntity())) || 
			 (flag.equals(PROPERTY_FLAGS.PRESENTATION) && (pd.getPresentationProperty()==null || pd.getPresentationProperty().isBlank())) ||
			 (flag.equals(PROPERTY_FLAGS.IMAGE_RENDERING) && (pd.getImageProcessing()==null || !pd.getImageProcessing().equals("render"))) ||
			 (flag.equals(PROPERTY_FLAGS.EXTERNAL_IDENTIFIER) && !pd.isExternalIdentifier()) ) {
			return false;
		}
		
		return true;
	}
	
	private static PropertyDefinition getDefinition(List<PropertyDefinition> properties, String[] path) {
		if (properties!=null) {
			for (PropertyDefinition def : properties) {
				if (def.getName().equals(path[0])) {
					if (path.length==1) {
						return def;
					} else if (HierarchicalPropertyDefinition.class.isAssignableFrom(def.getClass())){
						return getDefinition(HierarchicalPropertyDefinition.class.cast(def).getProperties(), Arrays.copyOfRange(path, 1, path.length));
					}
				} 
			}
		}
		return null;
	}
}
