package de.uniba.minf.registry.model.validation.constraints;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import am.ik.yavi.core.CustomConstraint;
import de.uniba.minf.registry.model.PropertyValue;
import lombok.Data;

@Data
public class MultiplicityConstraint<T> implements CustomConstraint<List<T>> {
	private final Integer min;
	private final Integer max;
	
	public static <T> MultiplicityConstraint<T> fromMin(int min) {
		return new MultiplicityConstraint<>(min, null);
	}
	
	public static <T> MultiplicityConstraint<T> fromMax(int max) {
		return new MultiplicityConstraint<>(null, max);
	}
	
	public static <T> MultiplicityConstraint<T> fromMinAndMax(int min, int max) {
		return new MultiplicityConstraint<>(min, max);
	}
	
	private MultiplicityConstraint(Integer min, Integer max) {
		this.min = min;
		this.max = max;
	}
	
    @Override
    public boolean test(List<T> c) {
        return ( (c==null || c.isEmpty()) && (min==null || min.intValue()==0)) // c empty and no minimum 
        		  ||
        		 (c!=null && !c.isEmpty() && // c not empty AND
        		 	(min==null || c.size()>=min.intValue()) && // gte minimum -- if any
        		 	(max==null || this.checkLteMax(c, max.intValue()))    // lte maximum -- if any
        		 );
    }
    
    // This counts values per language, as the max applies to each language
    private boolean checkLteMax(List<T> c, int max) {
    	if (c.size()<=max) {
    		return true;
    	}
    	
    	Map<Object, List<Object>> perKeyMap = new HashMap<>();
		for (T value : c) {
			if (PropertyValue.class.isAssignableFrom(value.getClass())) {
				PropertyValue pv = PropertyValue.class.cast(value);
				pv.valuesAsList().stream().forEach(v -> this.putOrAdd(pv.getLang(), v, perKeyMap));
			} else {
				this.putOrAdd(null, value, perKeyMap);
			}
		}
		
		for (List<Object> valuesPerKey : perKeyMap.values()) {
			if (valuesPerKey.size()>max) {
				return false;
			}
		}
    	
    	return true;
    }
    
    public void putOrAdd(Object key, Object value, Map<Object, List<Object>> c) {
    	List<Object> v;
    	if (c.containsKey(key)) {
    		v = c.get(key);
    	} else {
    		v = new ArrayList<>();
    		c.put(key, v);
    	}
    	v.add(value);
    }

    @Override
    public String messageKey() {
    	if (min==null && max==null) {
    		// both == null is not possible with applied singleton pattern
    		return null;
    	}
    	if (min==null) {
    		return "property.multiplicity.max";
    	} else if (max==null) {
    		return "property.multiplicity.min";
        } else if (min.equals(max)) {
    		return "property.multiplicity.fixed";
        } else if (min==0 && max==1) {
    		return "property.multiplicity.none";
        } else { 
        	return "property.multiplicity.minmax";
        }
    }

    @Override
    public String defaultMessageFormat() {
    	if (min==null && max==null) {
    		// both == null is not possible with applied singleton pattern
    		return null;
    	}
    	if (min==null) {
    		return "\"{0}\" has a specified multiplicity of max {2}";
    	} else if (max==null) {
    		return "\"{0}\" has a specified multiplicity of min {1}";
        } else if (min.equals(max)) {
    		return "\"{0}\" has a specified multiplicity of exactly {2}";
        } else if (min==0 && max==1) {
    		return "\"{0}\" does not allow multiplicity";
        } else { 
        	return "\"{0}\" has a specified multiplicity of min {1} and max {2}";
        }
    }
    
    @Override
    public Object[] arguments(List<T> violatedValue) {
    	return new Object[] {min, max};
    }
}