package de.uniba.minf.registry.model.serialization.base;

import static de.uniba.minf.registry.RegistryConstants.SERIALIZATION_CREATION_FIELD;
import static de.uniba.minf.registry.RegistryConstants.SERIALIZATION_DEFINITION_NAME_FIELD;
import static de.uniba.minf.registry.RegistryConstants.SERIALIZATION_DEFINITION_VERSION_FIELD;
import static de.uniba.minf.registry.RegistryConstants.SERIALIZATION_DELETED_FIELD;
import static de.uniba.minf.registry.RegistryConstants.SERIALIZATION_DRAFT_FIELD;
import static de.uniba.minf.registry.RegistryConstants.SERIALIZATION_ENTITYID_FIELD;
import static de.uniba.minf.registry.RegistryConstants.SERIALIZATION_INNER_LANG_FIELD;
import static de.uniba.minf.registry.RegistryConstants.SERIALIZATION_INNER_REFERENCE_FIELD;
import static de.uniba.minf.registry.RegistryConstants.SERIALIZATION_INNER_VALUE_FIELD;
import static de.uniba.minf.registry.RegistryConstants.SERIALIZATION_NEXT_VERSION_FIELD;
import static de.uniba.minf.registry.RegistryConstants.SERIALIZATION_PROPERTIES_FIELD;
import static de.uniba.minf.registry.RegistryConstants.SERIALIZATION_PUBLISHED_FIELD;
import static de.uniba.minf.registry.RegistryConstants.SERIALIZATION_READONLY_FIELD;
import static de.uniba.minf.registry.RegistryConstants.SERIALIZATION_SOURCE_ID_FIELD;
import static de.uniba.minf.registry.RegistryConstants.SERIALIZATION_SOURCE_LABEL_FIELD;
import static de.uniba.minf.registry.RegistryConstants.SERIALIZATION_TEMPLATE_FIELD;
import static de.uniba.minf.registry.RegistryConstants.SERIALIZATION_UNIQUEID_FIELD;

import java.io.IOException;
import java.time.Instant;
import java.time.format.DateTimeParseException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map.Entry;

import com.fasterxml.jackson.core.JacksonException;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.deser.std.StdDeserializer;

import de.uniba.minf.registry.model.BaseDefinedObject;
import de.uniba.minf.registry.model.BasePropertyValue;
import de.uniba.minf.registry.model.BooleanPropertyValue;
import de.uniba.minf.registry.model.DoublePropertyValue;
import de.uniba.minf.registry.model.IntegerPropertyValue;
import de.uniba.minf.registry.model.Property;
import de.uniba.minf.registry.model.PropertyImpl;
import de.uniba.minf.registry.model.PropertyList;
import de.uniba.minf.registry.model.PropertyValue;
import de.uniba.minf.registry.model.PropertyValueList;
import de.uniba.minf.registry.model.TextPropertyValue;
import de.uniba.minf.registry.model.entity.Entity;
import de.uniba.minf.registry.model.vocabulary.VocabularyEntry;
import lombok.extern.slf4j.Slf4j;

@Slf4j
public abstract class BaseDefinedObjectDeserializer<T extends BaseDefinedObject> extends StdDeserializer<T> {
	private static final long serialVersionUID = -873181459316190280L;

	protected BaseDefinedObjectDeserializer(Class<?> c) {
		super(c);
	}
	
	public void deserialize(T obj, JsonNode node) throws IOException, JacksonException {
	
		obj.setUniqueId(this.getTextValueIfExists(SERIALIZATION_UNIQUEID_FIELD, node));
		obj.setDefinitionName(this.getTextValueIfExists(SERIALIZATION_DEFINITION_NAME_FIELD, node));
		obj.setDefinitionVersion(this.getIntValue(SERIALIZATION_DEFINITION_VERSION_FIELD, node));
		
		obj.setReadOnly(this.getBooleanValue(SERIALIZATION_READONLY_FIELD, node));
		obj.setPublished(this.getBooleanValue(SERIALIZATION_PUBLISHED_FIELD, node));
		obj.setDraft(this.getBooleanValue(SERIALIZATION_DRAFT_FIELD, node));
		obj.setTemplate(this.getBooleanValue(SERIALIZATION_TEMPLATE_FIELD, node));
		obj.setDeleted(this.getBooleanValue(SERIALIZATION_DELETED_FIELD, node));
		
		obj.setSourceEntityId(this.getTextValueIfExists(SERIALIZATION_SOURCE_ID_FIELD, node));
		obj.setSourceLabel(this.getTextValueIfExists(SERIALIZATION_SOURCE_LABEL_FIELD, node));
				
		String creation = this.getTextValueIfExists(SERIALIZATION_CREATION_FIELD, node);
		if (creation!=null) {
			try {
				obj.setCreationInstant(Instant.parse(creation));
			} catch(DateTimeParseException e) {
				log.error("Failed to parse creation time with selected formatter", e);
			}
		}
		
		this.deserializeProperties(obj, node);
		this.deserializeLayers(obj, node);	
	}
	
	private void deserializeProperties(T obj, JsonNode node) {
		obj.setProperties(new ArrayList<>());
		if (!node.has(SERIALIZATION_PROPERTIES_FIELD)) {
			return;
		}
		
		Iterator<Entry<String, JsonNode>> propertyFieldIterator = node.get(SERIALIZATION_PROPERTIES_FIELD).fields();
		this.collectProperties(propertyFieldIterator, obj, node);
	}
	
	private void deserializeLayers(T entity, JsonNode node) {
		
	}
	
	private void collectProperties(Iterator<Entry<String, JsonNode>> propertyFieldIterator, PropertyList pl, JsonNode node) {
		Entry<String, JsonNode> propertyEntry;
		Property p;
		List<PropertyValue> propertyValues;
		
		while(propertyFieldIterator.hasNext()) {
			propertyEntry = propertyFieldIterator.next();
			p = new PropertyImpl();
			p.setLabel(propertyEntry.getKey());
			
			propertyValues = null;
			
			if (propertyEntry.getValue().isArray() && !propertyEntry.getValue().isEmpty()) {
				if (!propertyEntry.getValue().get(0).isObject() || propertyEntry.getValue().get(0).has(SERIALIZATION_INNER_VALUE_FIELD) || propertyEntry.getValue().get(0).has(SERIALIZATION_INNER_REFERENCE_FIELD)) {
					propertyValues = this.collectValues(propertyEntry.getValue());
				} else {
					p.setProperties(new ArrayList<>());
					this.collectHierarchicalPropertyLists(p.getProperties(), propertyEntry.getValue());
				}
			} else if (propertyEntry.getValue().isObject()) {
				if (propertyEntry.getValue().has(SERIALIZATION_INNER_VALUE_FIELD) || propertyEntry.getValue().has(SERIALIZATION_INNER_REFERENCE_FIELD)) {
					propertyValues = this.collectValues(propertyEntry.getValue());
				} else {
					p.setProperties(new ArrayList<>());
					this.collectHierarchicalPropertyLists(p.getProperties(), propertyEntry.getValue());
				}
			} else {
				propertyValues = this.collectValues(propertyEntry.getValue());		
			}
			
			if (propertyValues!=null && !propertyValues.isEmpty()) {
				if (propertyValues.size()==1) {
					p.setValue(propertyValues.get(0));
				} else {
					p.setValue(new PropertyValueList(propertyValues));
				}
			}
			
			if (!p.valuesAsList().isEmpty() || (p.getProperties()!=null && !p.getProperties().isEmpty())) {
				pl.add(p);
			}
		}
	}
	
	private void collectHierarchicalPropertyLists(List<PropertyList> pl, JsonNode node) {
		if (node.isArray()) {
			for (JsonNode plNode : node) {
				pl.add(this.collectHierarchicalProperties(plNode));
			}
		} else {
			pl.add(this.collectHierarchicalProperties(node));
		}
	}
	
	
	private PropertyList collectHierarchicalProperties(JsonNode node) {
		PropertyList pl = new PropertyList();
		Iterator<Entry<String, JsonNode>> propertyFieldIterator = node.fields();
		this.collectProperties(propertyFieldIterator, pl, node);
		return pl;
	}
	
	private List<PropertyValue> collectValues(JsonNode node) {
		List<PropertyValue> values = new ArrayList<>();
		BasePropertyValue<?> pv;
		if (node.isValueNode()) {
			pv = getPropertyValue(node);
			if (pv!=null) {
				values.add(pv);
			}
		} else if (node.isObject()) {
			if (node.has(SERIALIZATION_INNER_REFERENCE_FIELD)) {
				pv = getPropertyValue(node.get(SERIALIZATION_INNER_REFERENCE_FIELD));
			} else {
				pv = getPropertyValue(node.get(SERIALIZATION_INNER_VALUE_FIELD));
			}
			if (pv!=null) {
				JsonNode langNode = node.get(SERIALIZATION_INNER_LANG_FIELD);
				if (langNode!= null && !langNode.isMissingNode() && !langNode.isNull()) {
					pv.setLang(node.get(SERIALIZATION_INNER_LANG_FIELD).asText());
				}
				values.add(pv);
			}
		} else {
			// Iterate the arraynode
			for (JsonNode v : node) {
				values.addAll(this.collectValues(v));
			}
		}
		return values;
	}
	
	private BasePropertyValue<?> getPropertyValue(JsonNode node) {
		if (node==null || node.isMissingNode()) {
			return null;
		} else if (node.isBoolean()) {
			return new BooleanPropertyValue(node.asBoolean());
		} else if (node.isDouble()) {
			return new DoublePropertyValue(node.asDouble());
		} else if (node.isInt()) {
			return new IntegerPropertyValue(node.asInt());
		} else {
			return new TextPropertyValue(node.asText());
		}
	}
	
	
	protected String getTextValueIfExists(String field, JsonNode node) {
		if (node.has(field)) {
			return node.get(field).asText();
		}
		return null;
	}
	
	protected boolean getBooleanValue(String field, JsonNode node) {
		if (node.has(field)) {
			return node.get(field).asBoolean();
		}
		return false;
	}
	
	protected int getIntValue(String field, JsonNode node) {
		if (node.has(field)) {
			return node.get(field).asInt();
		}
		return 0;
	}
	
	protected double getDoubleValue(String field, JsonNode node) {
		if (node.has(field)) {
			return node.get(field).asDouble();
		}
		return 0;
	}
	
	protected long getLongValue(String field, JsonNode node) {
		if (node.has(field)) {
			return node.get(field).asLong();
		}
		return 0;
	}
}
