package de.uniba.minf.registry.model.helper;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.TextNode;

import de.uniba.minf.registry.model.BaseDefinedObject;
import de.uniba.minf.registry.model.BooleanPropertyValue;
import de.uniba.minf.registry.model.DoublePropertyValue;
import de.uniba.minf.registry.model.IntegerPropertyValue;
import de.uniba.minf.registry.model.Property;
import de.uniba.minf.registry.model.PropertyList;
import de.uniba.minf.registry.model.PropertyValue;
import de.uniba.minf.registry.model.PropertyValueList;
import de.uniba.minf.registry.model.TextPropertyValue;
import de.uniba.minf.registry.model.definition.BaseDefinition;
import de.uniba.minf.registry.model.definition.HierarchicalPropertyDefinition;
import de.uniba.minf.registry.model.definition.PropertyDefinition;
import de.uniba.minf.registry.model.definition.SimplePropertyDefinition;
import de.uniba.minf.registry.model.definition.SimplePropertyDefinition.SIMPLE_TYPES;
import de.uniba.minf.registry.model.entity.Entity;
import lombok.AllArgsConstructor;
import lombok.Data;

public class PropertyDefinitionHelper {
	private PropertyDefinitionHelper() {}
	
	@Data
	@AllArgsConstructor
	public static class UnknownProperty {
		private String path;
		private String label;
		private String value;
	}
	
	public static List<UnknownProperty> mergeWithDefinition(BaseDefinedObject e, BaseDefinition ed, boolean removeUnknown) {
		e.setDefinitionName(ed.getName());
		e.setDefinitionVersion(ed.getVersion());
		e.setPropertyDefinitions(ed.getAllProperties());

		List<UnknownProperty> unknownProperties = new ArrayList<>();
		
		mergePropertiesWithDefinitions(ed.getName(), e.getProperties(), ed.getAllProperties(), removeUnknown, unknownProperties);
		
		return unknownProperties;
	}
	
	private static void mergePropertiesWithDefinitions(String path, List<Property> ps, List<PropertyDefinition> pds, boolean removeUnknown, List<UnknownProperty> unknownProperties) {
		if (ps!=null) {
			List<Property> removeProperties = new ArrayList<>();
			ps.stream().forEach(p -> { 
				if (!mergePropertyWithDefinitions(path, p, pds==null ? Optional.empty() : pds.stream().filter(pd -> pd.getName().equals(p.getLabel())).findFirst(), removeUnknown, unknownProperties)) {
					removeProperties.add(p);
				}
			});
			if (removeUnknown) {
				ps.removeAll(removeProperties);
			}
		}
	}
	
	private static void mergePropertyListsWithDefinitions(String path, List<PropertyList> pls, List<PropertyDefinition> pds, boolean removeUnknown, List<UnknownProperty> unknownProperties) {
		pls.stream().filter(pl -> pl!=null).forEach(pl -> {
			pl.setPropertyDefinitions(pds);
			mergePropertiesWithDefinitions(path, pl.getProperties(), pds, removeUnknown, unknownProperties);
		});
	}
	
	private static boolean mergePropertyWithDefinitions(String path, Property p, Optional<PropertyDefinition> pd, boolean removeUnknown, List<UnknownProperty> unknownProperties) {
		String identifier = path + "." + p.getLabel();
		boolean result = true;
		if (pd.isEmpty()) {
			unknownProperties.add(new UnknownProperty(identifier, p.getLabel(), p.getValue()==null ? null : p.getValue().asText()));
			//log.warn("Skipped merging unmatched property '{}'", identifier);
			result = false;
		} else {
			p.setDefinition(pd.get());
			if ((pd.get().isSimple() || pd.get().isVocabulary()) && !pd.get().isMultilingual()) {
				p.valuesAsList().forEach(v -> v.setLang(null));
			}			
		}
				
		if (p.getProperties()!=null && pd.isPresent() && HierarchicalPropertyDefinition.class.isAssignableFrom(pd.get().getClass())) {
			mergePropertyListsWithDefinitions(identifier, p.getProperties(), HierarchicalPropertyDefinition.class.cast(pd.get()).getProperties(), removeUnknown, unknownProperties);
		} else if (p.getProperties()!=null) {
			mergePropertyListsWithDefinitions(identifier, p.getProperties(), new ArrayList<>(0), removeUnknown, unknownProperties);
		}
		return result;
	}

	public static void removeDefinitionsFromEntities(List<Entity> entities) {
		entities.stream().forEach(PropertyDefinitionHelper::removeDefinitionsFromEntity);
	}
	
	public static void removeDefinitionsFromEntity(Entity entity) {
		entity.setPropertyDefinitions(null);
		removeDefinitionsFromProperties(entity.getProperties());
	}
	
	public static void removeDefinitionsFromProperties(List<Property> properties) {
		if (properties==null) {
			return;
		}
		properties.stream().forEach(p -> { 
			p.setDefinition(null);
			removeDefinitionsFromPropertyLists(p.getProperties());
		});
	}
	
	public static void removeDefinitionsFromPropertyLists(List<PropertyList> propertyLists) {
		if (propertyLists==null) {
			return;
		}
		propertyLists.stream().forEach(p -> { 
			p.setPropertyDefinitions(null);
			removeDefinitionsFromProperties(p.getProperties());
		});
	}	
	
	public static PropertyValue transformIfRequiredAndPossible(PropertyDefinition definition, PropertyValue value) {
		// Vocabulary is always String -> no conversion
		if (value==null || definition==null || !definition.isSimple()) {
			return value;
		}
		
		SimplePropertyDefinition spd = SimplePropertyDefinition.class.cast(definition);
		if (PropertyValueList.class.isAssignableFrom(value.getClass())) {
			PropertyValueList result = new PropertyValueList();
			for (PropertyValue v : PropertyValueList.class.cast(value).getValues()) {
				result.add(PropertyDefinitionHelper.transformIfRequiredAndPossible(definition, v));
			}
			return result;
		} else if (spd.getType().equals(SIMPLE_TYPES.BOOLEAN) && !BooleanPropertyValue.class.isAssignableFrom(value.getClass())) {
			return new BooleanPropertyValue(value.asText().equalsIgnoreCase("true") || value.asText().equalsIgnoreCase("1"), value.getLang());
		} else if (spd.getType().equals(SIMPLE_TYPES.FLOAT) && !DoublePropertyValue.class.isAssignableFrom(value.getClass())) {
			try {
				return new DoublePropertyValue(Double.parseDouble(value.asText()), value.getLang());
			} catch (NumberFormatException e) {
				return value; // This will then fail validation
			}
		} else if (spd.getType().equals(SIMPLE_TYPES.INT) && !IntegerPropertyValue.class.isAssignableFrom(value.getClass())) {
			try {
				return new IntegerPropertyValue(Integer.parseInt(value.asText()), value.getLang());
			} catch (NumberFormatException e) {
				return value; // This will then fail validation
			}
		} else {
			return value;
		}
	}


	
}
