package de.uniba.minf.registry.model.validation;

import static de.uniba.minf.registry.model.validation.ValidationConstants.*;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

import am.ik.yavi.builder.ValidatorBuilder;
import am.ik.yavi.constraint.CollectionConstraint;
import de.uniba.minf.registry.model.BasePropertyValue;
import de.uniba.minf.registry.model.BooleanPropertyValue;
import de.uniba.minf.registry.model.DoublePropertyValue;
import de.uniba.minf.registry.model.IntegerPropertyValue;
import de.uniba.minf.registry.model.MissingPropertyValue;
import de.uniba.minf.registry.model.Property;
import de.uniba.minf.registry.model.PropertyList;
import de.uniba.minf.registry.model.PropertyValue;
import de.uniba.minf.registry.model.PropertyValueList;
import de.uniba.minf.registry.model.TextPropertyValue;
import de.uniba.minf.registry.model.definition.HierarchicalPropertyDefinition;
import de.uniba.minf.registry.model.definition.PropertyDefinition;
import de.uniba.minf.registry.model.definition.SimplePropertyDefinition;
import de.uniba.minf.registry.model.definition.SimplePropertyDefinition.SIMPLE_TYPES;
import de.uniba.minf.registry.model.entity.AutoqueryEntityLookupService;
import de.uniba.minf.registry.model.definition.VocabularyPropertyDefinition;
import de.uniba.minf.registry.model.validation.ValidationConstants.ValidationMethods;
import de.uniba.minf.registry.model.validation.constraints.MandatoryConstraint;
import de.uniba.minf.registry.model.validation.constraints.MultiplicityConstraint;
import de.uniba.minf.registry.model.validation.exception.ValidationConfigurationException;
import de.uniba.minf.registry.model.vocabulary.ValidationEntityService;
import de.uniba.minf.registry.model.vocabulary.VocabularyLookupService;
import lombok.Data;
import lombok.Getter;

public class PropertyValidator extends ValidatorBuilder<Property> {	
	@Getter private final List<ValidationConfigurationException> configurationExceptions = new ArrayList<>();
	
	@Getter private final VocabularyLookupService vocabularyLookupService;
	@Getter private final ValidationEntityService entityService;
	@Getter private final AutoqueryEntityLookupService autoqueryLookupService;
	
	@Data
	private class MinMaxBoundary {
		private final Integer min;
		private final Integer max;
	}
	
	public PropertyValidator(PropertyDefinition pd, VocabularyLookupService vocabularyLookupService, AutoqueryEntityLookupService autoqueryLookupService, ValidationEntityService entityService, boolean skipCompleteness) {
		this.vocabularyLookupService = vocabularyLookupService;
		this.entityService = entityService;
		this.autoqueryLookupService = autoqueryLookupService;
		this.buildValidation(pd, skipCompleteness);
	}
	
	private void buildValidation(PropertyDefinition pd, boolean skipCompleteness) {
		try {
			if (pd.isHierarchical()) {
				this.buildHierarchicalPropertyValidation(pd, skipCompleteness);
			} else {
				// VocabularyPropertyDefinition and SimplePropertyDefinition are handled similar for now 
				this.buildSimplePropertyValidation(pd, skipCompleteness);
			}
		} catch (ValidationConfigurationException ex) {
			configurationExceptions.add(ex);
		}
	}
	

	private void buildHierarchicalPropertyValidation(PropertyDefinition pd, boolean skipCompleteness) throws ValidationConfigurationException {
		// multiplicity on subitems
		if (!skipCompleteness) {
			Optional<CollectionConstraint<Property, List<PropertyList>, PropertyList>> multiplicityRestriction = this.getMultiplicityRestriction(pd);
			if (multiplicityRestriction.isPresent()) {
				this.constraintOnCondition((property, constraintGroup) -> property.getProperties()!=null && !property.getProperties().isEmpty(), 
						p -> p._collection(Property::getProperties, pd.getName() + "." + VALIDATION_METADATA_MULTIPLICITY, c -> multiplicityRestriction.get()));
			}
		}
		
		// Mandatory subitems
		if (!skipCompleteness) {
			Optional<CollectionConstraint<Property, List<PropertyList>, PropertyList>> mandatoryRestriction = this.getMandatoryRestriction(pd);
			if (mandatoryRestriction.isPresent()) {
				this.constraintOnCondition((property, constraintGroup) -> property.getProperties()!=null && property.getProperties().isEmpty(), 
						p -> p._collection(Property::getProperties, pd.getName() + "." + VALIDATION_METADATA_MANDATORY, c -> mandatoryRestriction.get()));
			}
		}
		
		// Build validation for subitems
		ValidatorBuilder<PropertyList> propListValidatorBuilder = ValidatorBuilder.of(PropertyList.class);
		HierarchicalPropertyDefinition hpd = HierarchicalPropertyDefinition.class.cast(pd);
		if (hpd.getProperties()!=null) {
			for (PropertyDefinition subPd : hpd.getProperties()) {
				PropertyValidator pv = new PropertyValidator(subPd, vocabularyLookupService, getAutoqueryLookupService(), entityService, skipCompleteness);
				propListValidatorBuilder.nest(e->e.get(subPd.getName()), VALIDATION_METADATA_PROPERTIES, pv.build());
				this.configurationExceptions.addAll(pv.getConfigurationExceptions());
			}
		}
		// Assign subitem validation to property lists / subitems
		this.forEach(Property::getProperties, pd.getName(), propListValidatorBuilder.build());
	}
	
	private void buildSimplePropertyValidation(PropertyDefinition pd, boolean skipCompleteness) throws ValidationConfigurationException {
		// multiplicity on values
		if (!skipCompleteness) {
			Optional<CollectionConstraint<Property, List<PropertyValue>, PropertyValue>> multiplicityRestriction = this.getMultiplicityRestriction(pd);
			if (multiplicityRestriction.isPresent()) {
				this.constraintOnCondition((property, constraintGroup) -> !property.valuesAsList().isEmpty(), 
						p -> p._collection(Property::valuesAsList, pd.getName() + "." + VALIDATION_METADATA_MULTIPLICITY, c -> multiplicityRestriction.get()));
			}
		}
		
		// Mandatory value
		if (!skipCompleteness) {
			Optional<CollectionConstraint<Property, List<PropertyValue>, PropertyValue>> mandatoryRestriction = this.getMandatoryRestriction(pd);
			if (mandatoryRestriction.isPresent()) {
				this.constraintOnCondition((property, constraintGroup) -> property.valuesAsList().isEmpty(), 
						p -> p._collection(Property::valuesAsList, pd.getName() + "." + VALIDATION_METADATA_MANDATORY, c -> mandatoryRestriction.get()));
			}
		}
		
		
		// Vocabulary properties
		if (pd.isVocabulary()) {
			VocabularyPropertyDefinition vpd = VocabularyPropertyDefinition.class.cast(pd);
			ValidatorBuilder<PropertyValueList> valueListValidatorBuilder = ValidatorBuilder.of(PropertyValueList.class); 
			PropertyValueValidator<? extends BasePropertyValue<?>> propertyValueValidator = new PropertyValueValidator<>(TextPropertyValue.class, vpd, vocabularyLookupService, getAutoqueryLookupService(), entityService);
			
			this.appendValueValidation(vpd, valueListValidatorBuilder, propertyValueValidator);
		}
		
		// Simple properties
		if (pd.isSimple()) {
			SimplePropertyDefinition spd = SimplePropertyDefinition.class.cast(pd);
			
			if (pd.isValidated() || !spd.getType().equals(SIMPLE_TYPES.TEXT)) {
				ValidatorBuilder<PropertyValueList> valueListValidatorBuilder = ValidatorBuilder.of(PropertyValueList.class); 
				PropertyValueValidator<? extends BasePropertyValue<?>> propertyValueValidator;
			
				//
				// TODO: Implement dates validity checks
				//
				if (spd.getType().equals(SIMPLE_TYPES.INT) || spd.getType().equals(SIMPLE_TYPES.DATE)) {
					propertyValueValidator = new PropertyValueValidator<>(IntegerPropertyValue.class, spd);
					this.appendTypeValidation(IntegerPropertyValue.class, spd, valueListValidatorBuilder);
				} else if (spd.getType().equals(SIMPLE_TYPES.BOOLEAN)) {
					propertyValueValidator = new PropertyValueValidator<>(BooleanPropertyValue.class, spd);
					this.appendTypeValidation(BooleanPropertyValue.class, spd, valueListValidatorBuilder);
				} else if (spd.getType().equals(SIMPLE_TYPES.FLOAT)) {
					propertyValueValidator = new PropertyValueValidator<>(DoublePropertyValue.class, spd);
					this.appendTypeValidation(DoublePropertyValue.class, spd, valueListValidatorBuilder);
				} else {
					propertyValueValidator = new PropertyValueValidator<>(TextPropertyValue.class, spd);
				}
				
				this.appendValueValidation(spd, valueListValidatorBuilder, propertyValueValidator);
			}
		}
	}
	
	private void appendValueValidation(PropertyDefinition pd, ValidatorBuilder<PropertyValueList> valueListValidatorBuilder, PropertyValueValidator<? extends BasePropertyValue<?>> propertyValueValidator) {
		// Append value validation both on value and valueList
		this.appendPropertyValueValidation(pd, valueListValidatorBuilder, propertyValueValidator);
		
		// Nest valueList validation below this (property validation)
		this.constraintOnCondition((value, constraintGroup) -> value.getValue()!=null && PropertyValueList.class.isAssignableFrom(value.getValue().getClass()),
		        c -> c.nest(v -> PropertyValueList.class.cast(v.getValue()), VALIDATION_METADATA_PROPERTIES, valueListValidatorBuilder.build()));
	}
		
	private void appendTypeValidation(Class<?> checkClass, SimplePropertyDefinition spd, ValidatorBuilder<PropertyValueList> valueListValidatorBuilder) {
		final CustomViolationMessages violationMessage = findViolationMessage(checkClass);
				
		// Single value
		this.constraintOnCondition((value, constraintGroup) -> !MissingPropertyValue.class.isAssignableFrom(value.getValue().getClass()) && !PropertyValueList.class.isAssignableFrom(value.getValue().getClass()),
		        c -> c.constraintOnTarget(v -> checkClass.isAssignableFrom(v.getValue().getClass()), spd.getName() + "." + VALIDATION_METADATA_VALUE, violationMessage));
		// Value lists
		valueListValidatorBuilder.forEach(PropertyValueList::getValues, spd.getName(), 
				c -> c.constraintOnTarget(v -> checkClass.isAssignableFrom(v.getClass()), VALIDATION_METADATA_VALUE , violationMessage)
		);
	}
	
	private CustomViolationMessages findViolationMessage(Class<?> checkClass) {
		for (CustomViolationMessages iViolationMessage : CustomViolationMessages.values()) {
			if (iViolationMessage.key().equals("checkClass:" + checkClass.getSimpleName())) {
				return iViolationMessage;
			}
		}
		return null;
	}
	
	@SuppressWarnings({ "unchecked", "rawtypes" })
	private void appendPropertyValueValidation(PropertyDefinition pd, ValidatorBuilder<PropertyValueList> valueListValidatorBuilder, PropertyValueValidator propertyValueValidator) {
		// Single value
		this.constraintOnCondition((value, constraintGroup) -> value.getValue()!=null && propertyValueValidator.getValidatedClass().isAssignableFrom(value.getValue().getClass()),
		        c -> c.nest(v -> propertyValueValidator.getValidatedClass().cast(v.getValue()), pd.getName() + "." + VALIDATION_METADATA_VALUE, propertyValueValidator.getValidator()));
		// Value lists		
		valueListValidatorBuilder.forEach(PropertyValueList::getValues, pd.getName(), 
				c -> c.constraintOnCondition((value, constraintGroup) -> value!=null &&propertyValueValidator.getValidatedClass().isAssignableFrom(value.getClass()),
						c1 -> c1.nest(v -> propertyValueValidator.getValidatedClass().cast(v), VALIDATION_METADATA_VALUE, propertyValueValidator.getValidator()))
		);
	}
	
	
	private <T> Optional<CollectionConstraint<Property, List<T>, T>> getMandatoryRestriction(PropertyDefinition pd) throws ValidationConfigurationException {
		if (pd.isMandatory()) {
			return Optional.of(new CollectionConstraint<Property, List<T>, T>().predicate(new MandatoryConstraint<>()));
		} else {
			return Optional.empty();
		}
	}
	
	private <T> Optional<CollectionConstraint<Property, List<T>, T>> getMultiplicityRestriction(PropertyDefinition pd) throws ValidationConfigurationException {
		// Multiplicity allowed and not further restricted
		if (pd.getMultiplicity()!=null && pd.getMultiplicity().size()==1 && pd.getMultiplicity().contains(VALIDATION_PROPERTY_TRUE_VALUE)) {
			return Optional.empty();
		}		
		
		// No multiplicity allowed
		if (pd.getMultiplicity()==null || pd.getMultiplicity().isEmpty() || pd.getMultiplicity().contains(VALIDATION_PROPERTY_FALSE_VALUE)) {		
			return Optional.of(new CollectionConstraint<Property, List<T>, T>().predicate(MultiplicityConstraint.fromMinAndMax(0, 1)));
		}
		return this.collectCustomMultiplicityConstraints(pd);
	}
	
	private <T> Optional<CollectionConstraint<Property, List<T>, T>> collectCustomMultiplicityConstraints(PropertyDefinition pd) throws ValidationConfigurationException {
		CollectionConstraint<Property, List<T>, T> c = new CollectionConstraint<>();
		MinMaxBoundary minMax = this.getMinMaxForMultiplicity(pd);
		if (minMax.getMin()!=null && minMax.getMax()!=null) {
			c.predicate(MultiplicityConstraint.fromMinAndMax(minMax.getMin(), minMax.getMax()));
		} else if (minMax.getMin()!=null) {
			c.predicate(MultiplicityConstraint.fromMin(minMax.getMin()));
		} else if (minMax.getMax()!=null) {
			c.predicate(MultiplicityConstraint.fromMax(minMax.getMax()));
		}
		
		return Optional.of(c);
	}
	
	private MinMaxBoundary getMinMaxForMultiplicity(PropertyDefinition pd) throws ValidationConfigurationException {
		Integer min = null;
		Integer max = null;
		
		for (String multi : pd.getMultiplicity()) {
			NumParamConstraint nC = new NumParamConstraint(multi);
			ValidationMethods method = nC.getAndAssertValidationMethod();
			if (method.equals(ValidationMethods.MIN) && (min==null || min>nC.getNum())) {
					min = nC.getNum();
			} else if (method.equals(ValidationMethods.MAX) && (max==null || max>nC.getNum())) {
					max = nC.getNum();
			} else if (method.equals(ValidationMethods.FIXED)) {
				if (max==null || max>nC.getNum()) {
					max = nC.getNum();
				}
				if (min==null || min>nC.getNum()) {
					min = nC.getNum();
				}
			} else {
				throw new ValidationConfigurationException("Configured constraint method not implemented for multiplicity validation: " + nC.getMethod());
			}
		}
		
		return new MinMaxBoundary(min, max);
	}
}
