package de.uniba.minf.registry.model.serialization.base;

import static de.uniba.minf.registry.model.serialization.base.BaseDefinitionSerializationConstants.*;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.fasterxml.jackson.core.JacksonException;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.deser.std.StdDeserializer;

import de.uniba.minf.registry.RegistryConstants.PropertyCompositionFlags;
import de.uniba.minf.registry.RegistryConstants.PropertyCompositionMethods;
import de.uniba.minf.registry.model.definition.AutofillPropertyDefinition;
import de.uniba.minf.registry.model.definition.BaseDefinition;
import de.uniba.minf.registry.model.definition.HierarchicalPropertyDefinition;
import de.uniba.minf.registry.model.definition.PropertyDefinition;
import de.uniba.minf.registry.model.definition.PropertyDefinitionBlock;
import de.uniba.minf.registry.model.definition.RelationPropertyDefinition;
import de.uniba.minf.registry.model.definition.SimplePropertyDefinition;
import de.uniba.minf.registry.model.definition.VocabularyPropertyDefinition;
import de.uniba.minf.registry.model.definition.SimplePropertyDefinition.SIMPLE_TYPES;
import de.uniba.minf.registry.model.serialization.base.SerializationMessage.MESSAGE_TYPES;
import lombok.extern.slf4j.Slf4j;

@Slf4j
public abstract class BaseDefinitionDeserializer<T extends BaseDefinition> extends StdDeserializer<T> {
	private static final long serialVersionUID = 8050899480597850564L;


	
	protected BaseDefinitionDeserializer(Class<?> vc) {
		super(vc);
	}

	@Override
	public T deserialize(JsonParser p, DeserializationContext ctxt) throws IOException, JacksonException {
		JsonNode node = p.getCodec().readTree(p);
		
		T def = this.createDefinition(node);
		
		if (node.has(PROPERTY_NAME_FIELD)) {
			def.setLabel(node.get(PROPERTY_NAME_FIELD).asText());
		}
		if (node.has(PROPERTY_CODE_FIELD)) {
			def.setMessageCode(node.get(PROPERTY_CODE_FIELD).asText());
		}
		if (node.has(SYSTEM_ENTITY_FIELD)) {
			def.setSystem(node.get(SYSTEM_ENTITY_FIELD).asBoolean());
		}
				
		if (node.has("endpoint")) {
			JsonNode endpointNode = node.get("endpoint");
			if (endpointNode.has("url")) {
				def.setEndpointUrl(endpointNode.get("url").asText());
			}
			if (endpointNode.has("method")) {
				def.setEndpointMethod(endpointNode.get("method").asText());
			}
		}
		
		try {
			def.setPropertyBlocks(new ArrayList<>());
			
			Map<String, PropertyDefinition> propertyIdentifierDefinitionsMap = new HashMap<>();
			
			if (node.has(PROPERTIES_FIELD) && node.get(PROPERTIES_FIELD).isArray()) {	
				def.getPropertyBlocks().add(this.createPropertyDefinitionBlock(node, def, null, propertyIdentifierDefinitionsMap));
			}
			if (node.has(PROPERTY_BLOCKS_FIELD) && node.get(PROPERTY_BLOCKS_FIELD).isArray()) {
				JsonNode blockLabelNode;
				for (JsonNode blockNode : node.get(PROPERTY_BLOCKS_FIELD)) {
					blockLabelNode = blockNode.get(PROPERTY_NAME_FIELD);
					if (blockNode.has(PROPERTIES_FIELD) && blockNode.get(PROPERTIES_FIELD).isArray()) {	
						def.getPropertyBlocks().add(this.createPropertyDefinitionBlock(blockNode, def, blockLabelNode!=null && blockLabelNode.isTextual() ? blockLabelNode.asText() : null, propertyIdentifierDefinitionsMap));
					}
				}
			}
			
			if (node.has(PREVIEW_PROPERTIES_FIELD)) {
				def.setPreviewPropertyIdentifiers(new ArrayList<>());
				for (JsonNode prevNode : node.get(PREVIEW_PROPERTIES_FIELD)) {
					if (propertyIdentifierDefinitionsMap.containsKey(prevNode.asText())) {
						def.getPreviewPropertyIdentifiers().add(prevNode.asText());
					} else if (propertyIdentifierDefinitionsMap.containsKey(def.getName() + "." + prevNode.asText())) {
						def.getPreviewPropertyIdentifiers().add(def.getName() + "." + prevNode.asText());
					} else {
						log.warn("Unknown preview property found and skipped: {}", prevNode.asText());
					}
				}
			}
			
		} catch (SerializationError e) {
			throw new IOException("Failed to deserialize entity definition", e);
		}
		return def;
	}

	protected PropertyDefinitionBlock createPropertyDefinitionBlock(JsonNode node, T def, String blockName, Map<String, PropertyDefinition> propertyIdentifierDefinitionsMap) throws SerializationError {
		PropertyDefinitionBlock block = new PropertyDefinitionBlock();
		block.setProperties(new ArrayList<>());
		if (blockName!=null) {
			block.setName(blockName);
			block.setIdentifier(def.getName() + "." + blockName);
		}
		PropertyDefinition propDef;
		if (node.has(PROPERTIES_FIELD) && node.get(PROPERTIES_FIELD).isArray()) {	
			for (JsonNode prop : node.get(PROPERTIES_FIELD)) {
				propDef = deserializePropertyDefinition(def, prop, def.getName(), propertyIdentifierDefinitionsMap);
				if (propDef!=null) {
					block.getProperties().add(propDef);
					propertyIdentifierDefinitionsMap.put(propDef.getIdentifier(), propDef);
				}
			}
		}
		return block;
	}
	
	protected abstract T createDefinition(JsonNode node);
		
	private PropertyDefinition deserializePropertyDefinition(T def, JsonNode node, String path, Map<String, PropertyDefinition> propertyIdentifierDefinitionsMap) throws SerializationError {
		if (!this.checkHasField(def, node, FIELD_FIELD, false, MESSAGE_TYPES.ERROR)) {
			return null;
		}
		String label = node.get(FIELD_FIELD).textValue();
		path = path + "." + label;
		
		PropertyDefinition propDef;
		if (node.has(PROPERTIES_FIELD)) {
			propDef = this.deserializeHierarchicalPropertyDefinition(def, node, path, label, propertyIdentifierDefinitionsMap);
		} else if (node.has(VOCABULARY_ENTITY_FIELD)) {
			propDef = this.deserializeVocabularyPropertyDefinition(def, node, VOCABULARY_ENTITY_FIELD, true);
		} else if (node.has(VOCABULARY_FIELD)) {
			propDef = this.deserializeVocabularyPropertyDefinition(def, node, VOCABULARY_FIELD, false);
		} else if (node.has(ENTITY_RELATION_NAME)) {
			propDef = this.deserializeRelationPropertyDefinition(def, node, ENTITY_RELATION_NAME);
		} else {
			propDef = this.deserializeSimplePropertyDefinition(def, node);
		}
		this.deserializePropertyAttributes(propDef, node, path, label);
		return propDef;
	}
	
	private void deserializePropertyAttributes(PropertyDefinition propDef, JsonNode node, String path, String label) {
		if (propDef!=null) {
			propDef.setName(label);
			if (node.has(PROPERTY_CODE_FIELD)) {
				propDef.setMessageCode(node.get(PROPERTY_CODE_FIELD).asText());
			}
			propDef.setIdentifier(path);
			this.setMandatory(node, propDef);
			this.setExternalIdentifier(node, propDef);
			
			propDef.setValid(this.getStringArgs(node, PROPERTY_VALID_FIELD));
			propDef.setMultilang(this.getStringArgs(node, PROPERTY_MULTILANG_FIELD));
			propDef.setMultiplicity(this.getStringArgs(node, PROPERTY_MULTIPLE_FIELD));
			
			if (node.has(PROPERTY_COMPOSITION_METHOD)) {
				propDef.setCompositionMethod(PropertyCompositionMethods.valueOf(node.get(PROPERTY_COMPOSITION_METHOD).asText().toUpperCase()));
			}
			if (node.has(PROPERTY_COMPOSITION_FLAG)) {
				propDef.setCompositionFlag(PropertyCompositionFlags.valueOf(node.get(PROPERTY_COMPOSITION_FLAG).asText().toUpperCase()));
			}
		}
	}
	
	private void setMandatory(JsonNode node, PropertyDefinition propDef) {
		if (node.has(PROPERTY_MANDATORY_FIELD)) {
			if (node.get(PROPERTY_MANDATORY_FIELD).isBoolean()) {
				propDef.setMandatory(node.get(PROPERTY_MANDATORY_FIELD).booleanValue());
				propDef.setMandatoryGroup(null);
			} else {
				propDef.setMandatoryGroup(node.get(PROPERTY_MANDATORY_FIELD).asText());
				if (propDef.getMandatoryGroup().isBlank()) {
					propDef.setMandatory(false);
					propDef.setMandatoryGroup(null);
				} else {
					propDef.setMandatory(true);
				}
			}
		}
	}
	
	private List<String> getStringArgs(JsonNode node, String field) {
		List<String> args = new ArrayList<>();
		if (node.has(field)) {
			if (node.get(field).isValueNode()) {
				for (String v : node.get(field).asText().split(",")) {
					args.add(v.trim());
				}
			} else if (node.get(field).isArray()) {
				for (JsonNode v : node.get(field)) {
					args.add(v.asText().trim());
				}
			}
		}
		return args;
	}

	private HierarchicalPropertyDefinition deserializeHierarchicalPropertyDefinition(T def, JsonNode node, String path, String label, Map<String, PropertyDefinition> propertyIdentifierDefinitionsMap) throws SerializationError {
		if (!this.checkHasField(def, node, PROPERTIES_FIELD, false, MESSAGE_TYPES.ERROR)) {
			return null;
		}
		HierarchicalPropertyDefinition propDef = new HierarchicalPropertyDefinition();
		propDef.setProperties(new ArrayList<>());
		PropertyDefinition childPropDef;

		for (JsonNode prop : node.get(PROPERTIES_FIELD)) {
			childPropDef = deserializePropertyDefinition(def, prop, path, propertyIdentifierDefinitionsMap);
			if (childPropDef!=null) {
				propDef.getProperties().add(childPropDef);
				propertyIdentifierDefinitionsMap.put(childPropDef.getIdentifier(), childPropDef);
			}
		}
		return propDef;
	}
	
	private PropertyDefinition deserializeRelationPropertyDefinition(T def, JsonNode node, String field) throws SerializationError {
		if (!this.checkHasField(def, node, field, false, MESSAGE_TYPES.ERROR)) {
			return null;
		}
		RelationPropertyDefinition propDef = new RelationPropertyDefinition();
		propDef.setRelation(node.get(field).textValue());
		return propDef;
	}

	private VocabularyPropertyDefinition deserializeVocabularyPropertyDefinition(T def, JsonNode node, String field, boolean entity) throws SerializationError {
		if (!this.checkHasField(def, node, field, false, MESSAGE_TYPES.ERROR)) {
			return null;
		}
		VocabularyPropertyDefinition propDef = new VocabularyPropertyDefinition();
		propDef.setVocabulary(node.get(field).textValue());
		this.setStrictVocabulary(node, propDef);
		if (entity) {
			propDef.setQuery(true);		
		} else {
			this.setQueryVocabulary(node, propDef);		
		}
		
		this.setInvertibleVocabulary(node, propDef);
		this.setAutoqueryVocabulary(node, propDef);
		this.setDefaultEntry(node, propDef);
		
		List<String> autofillArgs = this.getStringArgs(node, PROPERTY_VOCABULARY_AUTOFILL_FIELD);
		if (!autofillArgs.isEmpty()) {
			List<AutofillPropertyDefinition> autofillPropertyMap = new ArrayList<>();
			for (String arg : autofillArgs) {
				if (arg.contains("=")) {
					autofillPropertyMap.add(new AutofillPropertyDefinition(arg.substring(0, arg.indexOf('=')), arg.substring(arg.indexOf('=')+1)));
				} else {
					autofillPropertyMap.add(new AutofillPropertyDefinition(arg, null));
				}
			}
			propDef.setAutofillProperties(autofillPropertyMap);
		}
		propDef.setEntity(entity);
		
		return propDef;
	}

	protected void setStrictVocabulary(JsonNode node, VocabularyPropertyDefinition propDef) {
		if (node.has(PROPERTY_VOCABULARY_STRICT_FIELD)) {
			if (node.get(PROPERTY_VOCABULARY_STRICT_FIELD).isBoolean()) {
				propDef.setStrict(node.get(PROPERTY_VOCABULARY_STRICT_FIELD).asBoolean());
			} else if (node.get(PROPERTY_VOCABULARY_STRICT_FIELD).textValue().equals("radio")) {
				propDef.setStrict(true);
				propDef.setRadio(true);
			}
		}
	}
	
	private void setQueryVocabulary(JsonNode node, VocabularyPropertyDefinition propDef) {
		if (node.has(PROPERTY_VOCABULARY_QUERY_FIELD)) {
			propDef.setQuery(node.get(PROPERTY_VOCABULARY_QUERY_FIELD).asBoolean());
		}
	}
	
	private void setInvertibleVocabulary(JsonNode node, VocabularyPropertyDefinition propDef) {
		if (node.has(PROPERTY_VOCABULARY_INVERTIBLE_FIELD)) {
			propDef.setInvertible(node.get(PROPERTY_VOCABULARY_INVERTIBLE_FIELD).asBoolean());
		}
	}
	
	private void setAutoqueryVocabulary(JsonNode node, VocabularyPropertyDefinition propDef) {
		if (node.has(PROPERTY_VOCABULARY_AUTOQUERY_FIELD)) {
			propDef.setAutoQuery(node.get(PROPERTY_VOCABULARY_AUTOQUERY_FIELD).asBoolean());
		}
	}
		
	private void setExternalIdentifier(JsonNode node, PropertyDefinition propDef) {
		if (node.has(PROPERTY_IDENTIFIER_FIELD)) {
			propDef.setExternalIdentifier(node.get(PROPERTY_IDENTIFIER_FIELD).asBoolean());
		}
	}
	
	private void setDefaultEntry(JsonNode node, VocabularyPropertyDefinition propDef) {
		if (node.has(PROPERTY_VOCABULARY_DEFAULT_FIELD)) {
			propDef.setDefaultEntry(node.get(PROPERTY_VOCABULARY_DEFAULT_FIELD).asText());
			if (propDef.getDefaultEntry().isBlank()) {
				propDef.setDefaultEntry(null);
			}
		}
	}

	private SimplePropertyDefinition deserializeSimplePropertyDefinition(T def, JsonNode node) throws SerializationError {
		SimplePropertyDefinition propDef = new SimplePropertyDefinition();
		SIMPLE_TYPES type = null;
		if (node.has("type") && node.get("type").isValueNode()) {
			type = SIMPLE_TYPES.valueOf(node.get("type").textValue().toUpperCase());
		}
		if (type==null) {
			def.addMessage(SerializationMessage.handleMessage(MESSAGE_TYPES.NOTICE, "Type on field %s unknown or missing, assuming TEXT", node.get(FIELD_FIELD).textValue()));
			type = SIMPLE_TYPES.TEXT;
		}
		propDef.setType(type);
		return propDef;
	}

	
	private boolean checkHasField(T def, JsonNode node, String name, boolean failOnHasField, MESSAGE_TYPES messageType) throws SerializationError {
		if (node.has(name)==failOnHasField) {
			String message = String.format("%s: %s", failOnHasField ? "Has invalid field" : "Missing field", name);
			def.addMessage(SerializationMessage.handleMessage(messageType, message));
			return false;
		}
		return true;
	}
}
