package de.uniba.minf.registry.model.definition;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

import org.springframework.data.annotation.Transient;

import com.fasterxml.jackson.annotation.JsonIgnore;

import de.uniba.minf.registry.RegistryConstants.PropertyCompositionFlags;
import de.uniba.minf.registry.RegistryConstants.PropertyCompositionMethods;
import lombok.Data;

@Data
public abstract class PropertyDefinition implements Serializable {
	private static final long serialVersionUID = 8006255947259478674L;
	
	private String name;
	private String messageCode;
	private String identifier;
	
	private boolean mandatory;
	private boolean externalIdentifier;
	private List<String> multilang; // true || "deu, eng ..."
	private String mandatoryGroup;
	private List<String> valid;
	private List<String> multiplicity;
	
	private boolean excludeFromQuery;
	private float queryBoost;
	
	private PropertyCompositionMethods compositionMethod = PropertyCompositionMethods.DEFAULT;
	private PropertyCompositionFlags compositionFlag = PropertyCompositionFlags.NONE;
	
	public boolean isSimple() {
		return this.getClass().equals(SimplePropertyDefinition.class);
	}
	
	public boolean isVocabulary() {
		return this.getClass().equals(VocabularyPropertyDefinition.class);
	}
	
	public boolean isRelation() {
		return this.getClass().equals(RelationPropertyDefinition.class);
	}
	
	public boolean isHierarchical() {
		return HierarchicalPropertyDefinition.class.isAssignableFrom(this.getClass());
	}
	
	public SimplePropertyDefinition asSimple() {
		return SimplePropertyDefinition.class.cast(this);
	}
	
	public VocabularyPropertyDefinition asVocabulary() {
		return VocabularyPropertyDefinition.class.cast(this);
	}
	
	public HierarchicalPropertyDefinition asHierarchical() {
		return HierarchicalPropertyDefinition.class.cast(this);
	}
	
	public RelationPropertyDefinition asRelation() {
		return RelationPropertyDefinition.class.cast(this);
	}
	
	public boolean isValidated() {
		return this.valid!=null && !this.valid.isEmpty();
	}
	
	public boolean isMultiple() {
		return this.multiplicity!=null && !this.multiplicity.isEmpty() && !(this.multiplicity.size()==1 && this.multiplicity.get(0).equals("false"));
	}
	
	public boolean isMultilingual() {
		return this.multilang!=null && !this.multilang.isEmpty() && !(this.multilang.size()==1 && this.multilang.get(0).equals("false"));
	}
	
	public boolean isSpecifiedLanguages() {
		return this.isMultilingual() && !(this.multilang.size()==1 && this.multilang.get(0).equals("true"));
	}
	
	public List<String> getAllowedLanguages() {
		if (!this.isSpecifiedLanguages()) {
			return new ArrayList<>(0);
		}
		return this.multilang.stream().filter(l-> !l.equals("true") && !l.equals("false")).toList();
	}
	
	public boolean isSystem() {
		return this.getName().startsWith("_");
	}
	
	@JsonIgnore
	@Transient
	public String getIdentifierWithoutEntity() {
		return this.identifier.substring(this.identifier.indexOf('.')+1);
	}
	
	public String getMessageCode() {
		if (messageCode!=null) {
			return messageCode;
		}
		if (identifier.startsWith("~")) {
			return identifier;
		}		
		return "~" + identifier;
	}
}
