package de.uniba.minf.registry.model;

import java.util.ArrayList;
import java.util.List;
import org.springframework.data.annotation.Transient;

import de.uniba.minf.registry.model.definition.PropertyDefinition;
import lombok.Data;
import lombok.NoArgsConstructor;

@Data
@NoArgsConstructor
public class PropertyImpl implements Property {
	private static final long serialVersionUID = 8048151962532351314L;
	
	private String label;
	
	@Transient
	private PropertyDefinition definition;
	
	private PropertyValue value;
	
	private List<PropertyList> properties;
	
	// Rather specific for collected vocabulary property values from different layers
	private List<String> propertyKeyOrder;

	public PropertyImpl(PropertyDefinition definition) {
		this.setDefinition(definition);
	}
	
	public PropertyList getProperty(int index) {
		if (properties==null || properties.size()<index+1) {
			return null;
		} else {
			return properties.get(index);
		}
	}
	
	@Override 
	public boolean isEmpty() { 
		return (this.value==null || this.valuesAsList().isEmpty()) && (this.properties==null || this.properties.isEmpty()); 
	}

	@Override
	public List<PropertyValue> valuesAsList() {
		return value==null ? new ArrayList<>(0) : value.valuesAsList().stream().filter(v->!MissingPropertyValue.class.isAssignableFrom(v.getClass()) && !BlockedPropertyValue.class.isAssignableFrom(v.getClass())).toList();
	}
	
	@Override
	public List<String> blockedValues() {
		return value==null ? new ArrayList<>(0) : value.valuesAsList().stream().filter(v->BlockedPropertyValue.class.isAssignableFrom(v.getClass())).map(v->v.asText()).toList();
	}
	
	@Override public boolean isMissing() { return false; }

	@Override
	public void removeValues(List<PropertyValue> removeValues) {
		if (value!=null && removeValues.contains(value)) {
			value = null;
		} else if (value!=null && PropertyValueList.class.isAssignableFrom(value.getClass())) {
			PropertyValueList.class.cast(value).getValues().removeAll(removeValues);
		}		
	}

	@Override
	public void removeValue(PropertyValue removeValue) {
		if (value!=null && value.equals(removeValue)) {
			value = null;
		} else if (value!=null && PropertyValueList.class.isAssignableFrom(value.getClass())) {
			PropertyValueList.class.cast(value).getValues().remove(removeValue);
		}
	}

	@Override
	public void addValue(PropertyValue value) {
		if (this.getValue()==null) {
			this.setValue(value);
		} else if (this.getValue().isMultivalue()) {
			PropertyValueList values = PropertyValueList.class.cast(this.getValue());
			for (PropertyValue v : value.valuesAsList()) {
				if (!values.getValues().contains(v)) {
					values.add(v);
				}
			}
		} else if (!this.getValue().equals(value)) {
			PropertyValueList pv = new PropertyValueList();
			pv.add(this.getValue());
			for (PropertyValue v : value.valuesAsList()) {
				if (!pv.getValues().contains(v)) {
					pv.add(v);
				}
			}
			this.setValue(pv);
		}
	}

	@Override
	public void replaceTextValue(String findVal, String replaceVal) {
		if (this.getValue()!=null) {
			if (this.getValue().isMultivalue()) {
				for (PropertyValue v : this.getValue().valuesAsList()) {
					if (TextPropertyValue.class.isAssignableFrom(v.getClass()) && v.asText().equals(findVal)) {
						TextPropertyValue.class.cast(v).setValue(replaceVal);
					}
				}
			} else if (TextPropertyValue.class.isAssignableFrom(this.getValue().getClass()) && this.getValue().asText().equals(findVal)) {
				TextPropertyValue.class.cast(this.getValue()).setValue(replaceVal);
			}
		}
	}
}
