package de.uniba.minf.registry.model.validation;

import static de.uniba.minf.registry.model.validation.ValidationConstants.*;

import java.util.ArrayList;
import java.util.List;

import am.ik.yavi.builder.ValidatorBuilder;
import de.uniba.minf.registry.model.definition.EntityDefinition;
import de.uniba.minf.registry.model.definition.PropertyDefinition;
import de.uniba.minf.registry.model.entity.AutoqueryEntityLookupService;
import de.uniba.minf.registry.model.entity.Entity;
import de.uniba.minf.registry.model.validation.exception.ValidationConfigurationException;
import de.uniba.minf.registry.model.vocabulary.ValidationEntityService;
import de.uniba.minf.registry.model.vocabulary.VocabularyLookupService;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;

@Slf4j
public class EntityValidator extends ValidatorBuilder<Entity> {
	@Getter private final List<ValidationConfigurationException> configurationExceptions = new ArrayList<>();
	
	@Getter private final VocabularyLookupService vocabularyLookupService;
	@Getter private final ValidationEntityService entityService;
	@Getter private final AutoqueryEntityLookupService autoqueryLookupService;
	
	public EntityValidator(EntityDefinition ed, VocabularyLookupService vocabularyLookupService, AutoqueryEntityLookupService autoqueryLookupService, ValidationEntityService entityService, boolean skipCompleteness) throws ValidationConfigurationException {
		this.vocabularyLookupService = vocabularyLookupService;
		this.entityService = entityService;
		this.autoqueryLookupService = autoqueryLookupService;
		this.buildValidation(ed, skipCompleteness);
		if (!this.getConfigurationExceptions().isEmpty()) {
			configurationExceptions.forEach(x -> log.error("Validation configuration error", x));
			throw new ValidationConfigurationException("Errors occurred processing validation configuration", configurationExceptions);
		}
	}
	
	private void buildValidation(EntityDefinition ed, boolean skipCompleteness) {
		// Entity level property validation
		this.constraint(Entity::getDefinitionName, "definitionName", c -> c.equalTo(ed.getName()))
			.constraintOnCondition((entity, constraintGroup) -> entity.getDefinitionVersion()>0,
				b -> b.constraint(Entity::getDefinitionVersion, "definitionVersion", c -> c.equalTo(ed.getVersion())))
			;

		// Validate uniqueness of identifier fields
		// TODO: This needs to be moved to property validation somehow to produce better validation results
		ed.getExternalIdentifierProperties().stream().forEach(exp -> {
			UniqueExternalIdentifierConstraint uqc = new UniqueExternalIdentifierConstraint(ed, exp, getVocabularyLookupService(), getAutoqueryLookupService(), entityService);
			this._object(e->e, exp.getName(), c -> c.predicate(uqc));		
		});
		
		// Navigate to individual property validation
		for (PropertyDefinition pd : ed.getAllProperties()) {
			PropertyValidator pv = new PropertyValidator(pd, vocabularyLookupService, autoqueryLookupService, entityService, skipCompleteness);
			this.nest(e->e.get(pd.getName()), VALIDATION_METADATA_PROPERTIES, pv.build());
			this.configurationExceptions.addAll(pv.getConfigurationExceptions());
		}
	}
}