package de.uniba.minf.registry.model.serialization.base;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map.Entry;
import java.util.Optional;

import com.fasterxml.jackson.core.JacksonException;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.deser.std.StdDeserializer;
import com.fasterxml.jackson.databind.node.ObjectNode;

import de.uniba.minf.registry.model.BooleanPropertyValue;
import de.uniba.minf.registry.model.DoublePropertyValue;
import de.uniba.minf.registry.model.IntegerPropertyValue;
import de.uniba.minf.registry.model.Property;
import de.uniba.minf.registry.model.PropertyImpl;
import de.uniba.minf.registry.model.PropertyList;
import de.uniba.minf.registry.model.PropertyValue;
import de.uniba.minf.registry.model.PropertyValueList;
import de.uniba.minf.registry.model.TextPropertyValue;
import de.uniba.minf.registry.model.definition.BaseDefinition;
import de.uniba.minf.registry.model.definition.HierarchicalPropertyDefinition;
import de.uniba.minf.registry.model.definition.PropertyDefinition;
import de.uniba.minf.registry.model.definition.PropertyDefinitionBlock;
import de.uniba.minf.registry.model.helper.PropertyDefinitionHelper;

public abstract class BasePropertyListDeserializer<T extends PropertyList, U extends BaseDefinition> extends StdDeserializer<T> {
	private static final long serialVersionUID = -6318915380659257038L;
	
	private static final String DIRECT_FIELD_PREFIX = "_";
	
	public BasePropertyListDeserializer(Class<?> c) {
		super(c);
	}

	@Override
	public T deserialize(JsonParser p, DeserializationContext ctxt) throws IOException, JacksonException {
		JsonNode node = p.getCodec().readTree(p);
		
		T obj = this.createObject(node);
		U def = this.getDefinition(obj);
		obj.setProperties(new ArrayList<>());
		
		try {
			obj.setProperties(this.deserializePropertyObject(node, def==null ? new ArrayList<>() : def.getAllProperties(), def==null ? null : def.getPropertyBlocks()).getProperties());
			this.finalizeObject(obj, node);
		} catch (SerializationError e) {
			throw new IOException("Failed to deserialize obj", e);
		}		
		return obj;
	}
	
	protected abstract U getDefinition(T obj);

	protected abstract T createObject(JsonNode node);
	
	protected void finalizeObject(T obj, JsonNode node) {}
	
	private Property deserializeProperty(String field, JsonNode node, PropertyDefinition def) throws SerializationError {
		List<PropertyDefinition> subDefs = def!=null && HierarchicalPropertyDefinition.class.isAssignableFrom(def.getClass()) ? HierarchicalPropertyDefinition.class.cast(def).getProperties() : new ArrayList<>(0);
		Property prop = new PropertyImpl();
		
		String fieldName;
		String lang = null;
		if (field.contains("#")) {
			fieldName = field.substring(0, field.indexOf("#"));
			lang = field.substring(field.indexOf("#")+1);
		} else {
			fieldName = field;
		}
				
		if (node.isArray() && !node.isEmpty()) {
			if (node.get(0).has("_value")) {
				for (JsonNode subNode : node) {
					// Special multilang object with _value and _lang fields
					if (subNode.has("_value")) {
						lang = subNode.has("_lang") ? subNode.get("_lang").asText(null) : null;
						prop.addValue(this.getValue(subNode.get("_value"), def, lang));
					} else {
						prop.setProperties(new ArrayList<>());
						prop.getProperties().add(this.deserializePropertyObject(node, subDefs, null));
					}
				}
			} else {
				prop.setValue(this.getValueList(node, def, lang));
				prop.setProperties(this.getPropertyObjectList(node, subDefs));
			}
		} else if (node.isValueNode()) {
			prop.setValue(this.getValue(node, def, lang));
		} else if (node.isObject() && !node.isEmpty()) {
			// Special multilang object with _value and _lang fields
			if (node.has("_value")) {
				lang = node.has("_lang") ? node.get("_lang").asText(null) : null;
				prop.setValue(this.getValue(node.get("_value"), def, lang));
			} else {
				prop.setProperties(new ArrayList<>());
				prop.getProperties().add(this.deserializePropertyObject(node, subDefs, null));
			}
		} else {
			// Empty objects and array
			return null;
		}
		prop.setLabel(fieldName);
		return prop;
	}
	
	private List<PropertyList> getPropertyObjectList(JsonNode array, List<PropertyDefinition> defs) throws SerializationError {		
		List<PropertyList> list = new ArrayList<>();
		for (JsonNode node : array) {
			if (node.isObject()) {
				list.add(this.deserializePropertyObject(node, defs, null));
			}
		}
		return list.isEmpty() ? null : list;
	}	
	
	private PropertyList deserializePropertyObject(JsonNode node, List<PropertyDefinition> defs, List<PropertyDefinitionBlock> blocks) throws SerializationError {
		PropertyList properties = new PropertyList();	
		Iterator<Entry<String, JsonNode>> fieldIterator = node.fields();
		Property property;
		Optional<PropertyDefinition> def;
		while (fieldIterator.hasNext()) {
			Entry<String, JsonNode> fieldEntry = fieldIterator.next();
			if (fieldEntry.getKey().startsWith(DIRECT_FIELD_PREFIX)) {
				continue;
			}
			def = defs.stream().filter(d->d.getName().equals(fieldEntry.getKey())).findFirst();
			
			// No immediate property definition matched, imported data could be prefixed with block label
			if (def.isEmpty() && blocks!=null) {
				Optional<PropertyDefinitionBlock> bl = blocks.stream().filter(b -> b.getName()!=null && b.getName().equals(fieldEntry.getKey())).findFirst();
				if (bl.isPresent()) {
					if (fieldEntry.getValue().isArray()) {
						List<PropertyList> pls = this.getPropertyObjectList(fieldEntry.getValue(), bl.get().getProperties());
						if (pls!=null && !pls.isEmpty()) {
							for (PropertyList pl : pls) {
								if (pl.getProperties()!=null && !pl.getProperties().isEmpty()) {
									pl.getProperties().stream().forEach(p -> properties.set(p.getLabel(), p.getValue(), true));
								}
							}
						}
					} else {
						PropertyList pl = this.deserializePropertyObject(fieldEntry.getValue(), bl.get().getProperties(), null);
						if (pl.getProperties()!=null && !pl.getProperties().isEmpty()) {
							pl.getProperties().stream().forEach(p -> properties.set(p.getLabel(), p.getValue(), true));
						}
					}
					// Property has been handled, continue  
					continue;
				}
			} 
			
			property = this.deserializeProperty(fieldEntry.getKey(), fieldEntry.getValue(), def.orElse(null));
			
			if (property!=null) {
				properties.add(property);
			}
		}
		return properties;
	}
	
	private PropertyValue getValue(JsonNode node, PropertyDefinition def, String lang) {		
		if (def==null) {
			if (node.isDouble()) {
				return new DoublePropertyValue(node.asDouble(), lang);
			}
			if (node.isBoolean()) {
				return new BooleanPropertyValue(node.asBoolean(), lang);
			}		
			if (node.canConvertToInt()) {
				return new IntegerPropertyValue(node.asInt(), lang);
			}
			// In case the language is encoded
			if (node.asText().length()>5 && node.asText().charAt(3)=='#') {
				return new TextPropertyValue(node.asText().substring(4), node.asText().substring(0, 3));
			} else {
				return new TextPropertyValue(node.asText());
			}
		} else {
			return PropertyDefinitionHelper.transformJsonTree(def, node, lang); 
		}
	}
	
	private PropertyValueList getValueList(JsonNode array, PropertyDefinition def, String lang) {		
		PropertyValueList list = new PropertyValueList();
		for (JsonNode node : array) {
			if (node.isValueNode()) {
				list.add(this.getValue(node, def, lang));
			}
		}
		return list.isEmpty() ? null : list;
	}	
}
