package de.uniba.minf.registry.model.serialization.base;

import static de.uniba.minf.registry.model.serialization.base.BaseDefinitionSerializationConstants.*;

import java.io.IOException;
import java.util.List;

import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.databind.SerializerProvider;
import com.fasterxml.jackson.databind.ser.std.StdSerializer;

import de.uniba.minf.registry.model.definition.AutofillPropertyDefinition;
import de.uniba.minf.registry.model.definition.BaseDefinition;
import de.uniba.minf.registry.model.definition.HierarchicalPropertyDefinition;
import de.uniba.minf.registry.model.definition.PropertyDefinition;
import de.uniba.minf.registry.model.definition.PropertyDefinitionBlock;
import de.uniba.minf.registry.model.definition.SimplePropertyDefinition;
import de.uniba.minf.registry.model.definition.VocabularyPropertyDefinition;

public abstract class BaseDefinitionSerializer<T extends BaseDefinition> extends StdSerializer<T> {
	private static final long serialVersionUID = 4058774142036126961L;

	protected BaseDefinitionSerializer(Class<T> vc) {
		super(vc);
	}

	@Override
	public void serialize(T d, JsonGenerator gen, SerializerProvider provider) throws IOException {
		this.writeStringField(PROPERTY_NAME_FIELD, d.getLabel(), gen);
		this.writeTrueField(SYSTEM_ENTITY_FIELD, d.isSystem(), gen);
        this.writeMessageCode(d.getMessageCode(), d.getName(), d.getName(), gen);
        
        if (d.getEndpointUrl()!=null || d.getEndpointMethod()!=null) {
        	gen.writeObjectFieldStart("endpoint");
        	this.writeStringField("url", d.getEndpointUrl(), gen);
        	this.writeStringField("method", d.getEndpointMethod(), gen);
        	gen.writeEndObject();
        }
        
        if (d.getPreviewPropertyIdentifiers()!=null && !d.getPreviewPropertyIdentifiers().isEmpty()) {
        	gen.writeArrayFieldStart(PREVIEW_PROPERTIES_FIELD);
			for (String b : d.getPreviewPropertyIdentifiers()) {
				gen.writeString(b.substring(d.getName().length()+1));
			}
			gen.writeEndArray();
        }
        
        if (d.getPropertyBlocks()!=null) {
        	// Empty block -> write properties immediately
        	if (d.getPropertyBlocks().size()==1 && d.getPropertyBlocks().get(0).getName()==null) {
        		this.serializeProperties(d.getPropertyBlocks().get(0).getProperties(), gen, provider);
        	} else {
    			gen.writeArrayFieldStart(PROPERTY_BLOCKS_FIELD);
    			for (PropertyDefinitionBlock b : d.getPropertyBlocks()) {
    				gen.writeStartObject();
    				this.serializePropertyBlock(b, gen, provider);
    				gen.writeEndObject();
    			}
    			gen.writeEndArray();
        	}
        }
	}
	
	private void serializePropertyBlock(PropertyDefinitionBlock b, JsonGenerator gen, SerializerProvider provider) throws IOException {
		this.writeStringField(PROPERTY_NAME_FIELD, b.getName(), gen);
		this.writeMessageCode(b.getMessageCode(), b.getName(), b.getIdentifier(), gen);
		this.serializeProperties(b.getProperties(), gen, provider);
	}
		
	private void serializeProperties(List<PropertyDefinition> pds, JsonGenerator gen, SerializerProvider provider) throws IOException {
		if (pds==null || pds.isEmpty()) {
			return;
		}
		gen.writeArrayFieldStart(PROPERTIES_FIELD);
		for (PropertyDefinition pd : pds) {
			gen.writeStartObject();
			this.serializePropertyDefinition(pd, gen, provider);
			gen.writeEndObject();
		}
		gen.writeEndArray();
	}
	
	private void serializeVocabularyPropertyDefinition(VocabularyPropertyDefinition vpd, JsonGenerator gen, SerializerProvider provider) throws IOException {
		if (vpd.isEntity()) {
			this.writeStringField(VOCABULARY_ENTITY_FIELD, vpd.getVocabulary(), gen);
			this.writeStringField(VOCABULARY_ENTITY_RELATION_TYPE_FIELD, vpd.getEntityRelationType(), gen);
			this.writeStringField(VOCABULARY_ENTITY_RELATION_VOCABULARY_FIELD, vpd.getEntityRelationVocabulary(), gen);
		} else {
			this.writeStringField(VOCABULARY_FIELD, vpd.getVocabulary(), gen);
		}
		
		this.writeTrueField(PROPERTY_VOCABULARY_STRICT_FIELD, vpd.isStrict(), gen);
		this.writeTrueField(PROPERTY_VOCABULARY_QUERY_FIELD, vpd.isQuery(), gen);
		this.writeTrueField(PROPERTY_VOCABULARY_AUTOQUERY_FIELD, vpd.isAutoQuery(), gen);
		this.writeStringField(PROPERTY_VOCABULARY_DEFAULT_FIELD, vpd.getDefaultEntry(), gen);
	
		if (vpd.isAutofillAll()) {
			gen.writeBooleanField(PROPERTY_VOCABULARY_AUTOFILL_FIELD, true);
		} else if (vpd.isAutofill()) {
			gen.writeArrayFieldStart(PROPERTY_VOCABULARY_AUTOFILL_FIELD);
			for (AutofillPropertyDefinition afPd : vpd.getAutofillProperties()) {
				if (afPd.getExternalName()!=null) {
					gen.writeString(afPd.getPropertyName() + "=" + afPd.getExternalName());
				} else {
					gen.writeString(afPd.getPropertyName());
				}
			}
			gen.writeEndArray();
		}
	}
	
	private void serializePropertyDefinition(PropertyDefinition pd, JsonGenerator gen, SerializerProvider provider) throws IOException {
		gen.writeStringField(FIELD_FIELD, pd.getName());
		this.writeMessageCode(pd.getMessageCode(), pd.getName(), pd.getIdentifier(), gen);
		
		if (pd.isVocabulary()) {
			this.serializeVocabularyPropertyDefinition(VocabularyPropertyDefinition.class.cast(pd), gen, provider);
		} else if (pd.isSimple()) {
			this.writeStringField("type", SimplePropertyDefinition.class.cast(pd).getType().name().toLowerCase(), gen);
		}
		
		if (pd.getMandatoryGroup()!=null) {
			this.writeStringField(PROPERTY_MANDATORY_FIELD, pd.getMandatoryGroup(), gen);
		} else {
			this.writeTrueField(PROPERTY_MANDATORY_FIELD, pd.isMandatory(), gen);
		}
		
		this.writeTrueField(PROPERTY_IDENTIFIER_FIELD, pd.isExternalIdentifier(), gen);

		if (pd.getMultiplicity()!=null && !pd.getMultiplicity().isEmpty()) {
			this.writeStringArgs(PROPERTY_MULTIPLE_FIELD, pd.getMultiplicity(), gen);
		}
		
		if (pd.getMultilang()!=null && !pd.getMultilang().isEmpty()) {
			this.writeStringArgs(PROPERTY_MULTILANG_FIELD, pd.getMultilang(), gen);
		}
		
		if (pd.getValid()!=null && !pd.getValid().isEmpty()) {
			this.writeStringArgs(PROPERTY_VALID_FIELD, pd.getValid(), gen);
		}
		
		if (pd.isHierarchical()) {
			this.serializeProperties(HierarchicalPropertyDefinition.class.cast(pd).getProperties(), gen, provider);
		}
	}
	
	protected void writeStringArgs(String field, List<String> args, JsonGenerator gen) throws IOException {
		if (args.size()==1) {
			this.writeStringAsStringOrBooleanField(field, args.get(0), gen);
		} else {
			gen.writeArrayFieldStart(field);
			for (String v : args) {
				this.writeStringAsStringOrBoolean(v, gen);
			}
			gen.writeEndArray();
		}
	}
	
	protected void writeStringAsStringOrBoolean(String value, JsonGenerator gen) throws IOException {
		if (value.equalsIgnoreCase("false") || value.equalsIgnoreCase("true")) {
			gen.writeBoolean(value.toLowerCase().equals("true"));
		} else {
			gen.writeString(value);
		}
	}
	
	protected void writeStringAsStringOrBooleanField(String field, String value, JsonGenerator gen) throws IOException {
		if (value.equalsIgnoreCase("false") || value.equalsIgnoreCase("true")) {
			gen.writeBooleanField(field, value.toLowerCase().equals("true"));
		} else {
			gen.writeStringField(field, value);
		}
	}

	protected void writeTrueField(String field, boolean value, JsonGenerator gen) throws IOException {
		if (value) {
			gen.writeBooleanField(field, value);
		}
	}
	
	protected void writeStringField(String field, String value, JsonGenerator gen) throws IOException {
		if (value!=null) {
			gen.writeStringField(field, value);
		}
	}
	
	protected void writeMessageCode(String code, String label, String identifier, JsonGenerator gen) throws IOException {
		if (!code.equals("~" + label) && !code.equals("~" + identifier)) {
			gen.writeStringField(PROPERTY_CODE_FIELD, code);
		}
	}
}
