package de.uniba.minf.registry.model.definition;

import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;

import org.springframework.data.annotation.Id;
import org.springframework.data.mongodb.core.index.CompoundIndex;
import org.springframework.data.mongodb.core.index.Indexed;

import de.uniba.minf.core.rest.model.Identifiable;
import de.uniba.minf.registry.model.serialization.base.SerializationMessage;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;

@Slf4j
@Data
@CompoundIndex(def = "{'name': 1, 'version': 1}", unique = true)
public abstract class BaseDefinition implements Identifiable {
	private static final long serialVersionUID = 4908093821116664728L;

	private enum PROPERTY_FLAGS { AUTOFILL, AUTOQUERY, ENTITY_VOCABULARY, EXTERNAL_IDENTIFIER }
	
	/**
	 * ID of the version -> changes with every update
	 */
	@Id
	private String uniqueId;
	
	/**
	 * Identifier of the definition
	 */
	@Indexed
	private String name;
	
	/**
	 * Informal use only
	 */
	private long version;
	
	/**
	 * Creation instant of the version
	 */
	private Instant creationInstant;
	private String userUniqueId;	
	
	private String label;
	private String messageCode;
	
	private String endpointUrl;
	private String endpointMethod;
	
	@Indexed
	private String nextVersionUniqueId;
	
	private boolean system;
	
	public boolean isRemote() {
		return this.endpointUrl!=null;
	}
	
	public String getMessageCode() {
		return messageCode!=null ? messageCode : ("~" + name);
	}
	
	private List<SerializationMessage> messages;
	private List<PropertyDefinitionBlock> propertyBlocks;
	
	private List<String> previewPropertyIdentifiers;
	
	public List<PropertyDefinition> getPreviewProperties() {
		final List<PropertyDefinition> previewProperties = new ArrayList<>();
		if (previewPropertyIdentifiers!=null && !previewPropertyIdentifiers.isEmpty()) {
			for (String previewPropertyIdentifier : previewPropertyIdentifiers) {
				Optional<PropertyDefinition> prevPd = this.getAllProperties().stream().filter(p -> p.getIdentifier().equals(previewPropertyIdentifier)).findFirst();
				if (prevPd.isPresent()) {
					previewProperties.add(prevPd.get());
				} else {
					log.warn("Skipping unknown preview property identifier: " + previewPropertyIdentifier);
				}
			}
		}
		if (previewProperties.size()<5) {
			previewProperties.addAll(this.getAllProperties().stream().limit(5-previewProperties.size()).toList());
		}
		
		return previewProperties;
	}
	
	public List<PropertyDefinition> getOrderedProperties() {
		final List<PropertyDefinition> orderedProperties = new ArrayList<>();
		orderedProperties.addAll(this.getPreviewProperties());
		orderedProperties.addAll(this.getAllProperties().stream().filter(p -> !orderedProperties.contains(p) && !p.isSystem() && (p.isSimple() || p.isVocabulary())).toList());
		return orderedProperties;
	}
	
	public void addMessage(SerializationMessage message) {
		if (messages==null) {
			messages = new ArrayList<>();
		}
		messages.add(message);
	}
	
	public PropertyDefinition getDefinition(String identifier) {
		String[] path = identifier.split("\\.");
		return getDefinition(this.getAllProperties(), path);
	}
	
	public PropertyDefinition getDefinition(String... path) {
		return getDefinition(this.getAllProperties(), path);
	}
	
	public boolean hasAutoqueryProperties() {
		return !this.getAutoqueryProperties().isEmpty();
	}
	
	public boolean hasAutofillProperties() {
		return !this.getAutofillProperties().isEmpty();
	}
	
	public List<PropertyDefinition> getAllProperties() {
		return this.propertyBlocks.stream().flatMap(b -> b.getProperties().stream()).toList();
	}
	
	public List<VocabularyPropertyDefinition> getAutoqueryProperties() {
		return this.getProperties(this.getAllProperties(), PROPERTY_FLAGS.AUTOQUERY);
	}
	
	public List<VocabularyPropertyDefinition> getAutofillProperties() {
		return this.getProperties(this.getAllProperties(), PROPERTY_FLAGS.AUTOFILL);
	}
	
	public List<VocabularyPropertyDefinition> getEntityVocabularyProperties() {
		return this.getProperties(this.getAllProperties(), PROPERTY_FLAGS.ENTITY_VOCABULARY);
	}
	
	public List<PropertyDefinition> getExternalIdentifierProperties() {
		return this.getProperties(this.getAllProperties(), PROPERTY_FLAGS.EXTERNAL_IDENTIFIER);
	}
	
	private <T extends PropertyDefinition> List<T> getProperties(List<PropertyDefinition> properties, PROPERTY_FLAGS... flags) {
		List<T> matchingDefinitions = new ArrayList<>();
		for (PropertyDefinition pd : properties) {
			if (pd.isHierarchical()) {
				matchingDefinitions.addAll(this.getProperties(HierarchicalPropertyDefinition.class.cast(pd).getProperties(), flags));
			} else {
				if (this.getPropertyIsMatch(pd, flags)) {
					matchingDefinitions.add((T)pd);
				}
			}
		}
		return matchingDefinitions;
	}
	
	private boolean getPropertyIsMatch(PropertyDefinition pd, PROPERTY_FLAGS... flags) {
		for (PROPERTY_FLAGS flag : flags) {
			if ( (flag.equals(PROPERTY_FLAGS.AUTOQUERY) && (!pd.isVocabulary() || !VocabularyPropertyDefinition.class.cast(pd).isAutoQuery())) ||
				 (flag.equals(PROPERTY_FLAGS.AUTOFILL) && (!pd.isVocabulary() || !VocabularyPropertyDefinition.class.cast(pd).isAutofill())) ||	
				 (flag.equals(PROPERTY_FLAGS.ENTITY_VOCABULARY) && (!pd.isVocabulary() || !VocabularyPropertyDefinition.class.cast(pd).isEntity())) || 
				 (flag.equals(PROPERTY_FLAGS.EXTERNAL_IDENTIFIER) && !pd.isExternalIdentifier()) ) {
				return false;
			}
		}
		return true;
	}
	
	private static PropertyDefinition getDefinition(List<PropertyDefinition> properties, String[] path) {
		if (properties!=null) {
			for (PropertyDefinition def : properties) {
				if (def.getName().equals(path[0])) {
					if (path.length==1) {
						return def;
					} else if (HierarchicalPropertyDefinition.class.isAssignableFrom(def.getClass())){
						return getDefinition(HierarchicalPropertyDefinition.class.cast(def).getProperties(), Arrays.copyOfRange(path, 1, path.length));
					}
				} 
			}
		}
		return null;
	}
}
