package de.uniba.minf.registry.model;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.springframework.data.annotation.Transient;

import de.uniba.minf.registry.model.definition.HierarchicalPropertyDefinition;
import de.uniba.minf.registry.model.definition.PropertyDefinition;
import de.uniba.minf.registry.model.helper.PropertyDefinitionHelper;
import lombok.Data;
import lombok.NoArgsConstructor;

@Data
@NoArgsConstructor
public class PropertyList implements Serializable {
	private static final long serialVersionUID = -6867411678627272950L;
	
	@Transient
	private List<PropertyDefinition> propertyDefinitions;
	
	public PropertyList(List<PropertyDefinition> propertyDefinitions) {
		this.propertyDefinitions = propertyDefinitions;
	}
	
	@Data
	private class IndexedField {
		private static final Pattern INDEXED_FIELD_PATTERN = Pattern.compile("([^#\\[]+)(\\[([^\\]]+)\\])?(\\#([^\\[]+))?");
		
		private String fieldname = null;
		private Integer index = null;
		private String subpropSelector = null;
		
		public IndexedField(String name) {
			Matcher matcher = INDEXED_FIELD_PATTERN.matcher(name);
	       
	        if (matcher.find()) {	            
	            if (matcher.groupCount()>0) {
	            	fieldname = matcher.group(1);
	            }
	            if (matcher.groupCount()>2) {
	            	index = matcher.group(3)==null ? null : Integer.parseInt(matcher.group(3));
	            }
	            if (matcher.groupCount()>4) {
	            	subpropSelector = matcher.group(5);
	            }
	        }
		}
		
		public boolean hasIndex() {
			return this.index!=null;
		}
	}
	
	private List<Property> properties;
	
	public List<Property> get(PropertyDefinition propertyDefinition) {
		String[] pathWithEntity = propertyDefinition.getIdentifier().split("\\.");
		String[] getPath = Arrays.copyOfRange(pathWithEntity, 1, pathWithEntity.length);
		
		return this.getByIdentifier(getPath);
	}
	
	public List<Property> getByIdentifier(String[] identifier) {
		return this.getByIdentifierInProperties(this.getProperties(), identifier);
	}
	
	private List<Property> getByIdentifierInProperties(List<Property> properties, String[] identifier) {
		List<Property> result = new ArrayList<>();
		if (properties!=null) {
			for (Property prop : properties) {
				if (prop.getLabel().equals(identifier[0])) {
					if (identifier.length==1) {
						result.add(prop);
					} else {
						result.addAll(this.getByIdentifierInPropertyLists(prop.getProperties(), Arrays.copyOfRange(identifier, 1, identifier.length)));
					}
					
				}
			}
		}
		return result;
	}
	
	private List<Property> getByIdentifierInPropertyLists(List<PropertyList> propertyLists, String[] identifier) {
		return propertyLists.stream().flatMap(pl -> this.getByIdentifierInProperties(pl.getProperties(), identifier).stream()).toList();
	}
	
	public Property get(String property) {
		String[] path = property.split("\\.");
		return this.get(path);
	}
	
	public Property get(String[] path) {
		if (properties!=null) {
			for (Property prop : properties) {
				
				IndexedField idxF = new IndexedField(path[0]);
				if (prop.getLabel().equals(idxF.getFieldname())) {
					if (idxF.hasIndex()) {
						return prop.getProperty(idxF.getIndex()).get(Arrays.copyOfRange(path, 1, path.length));
					} else {
						return prop;
					}
				}
			}
		}
		return MissingProperty.getInstance();
	}
	
	public void set(String property, boolean value) {
		this.set(property, new BooleanPropertyValue(value));
	}

	public void set(String property, int value) {
		this.set(property, new IntegerPropertyValue(value));
	}
	
	public void set(String property, String value) {
		this.set(property, new TextPropertyValue(value));
	}
	
	public void set(String property, String value, boolean append) {
		this.set(property, new TextPropertyValue(value), append);
	}
	
	public void set(String property, double value) {
		this.set(property, new DoublePropertyValue(value));
	}
	
	public void set(String property, PropertyValue value) {
		String[] path = property.split("\\.");
		this.set(path, value);
	}
	
	public void set(String property, PropertyValue value, boolean append) {
		String[] path = property.split("\\.");
		this.set(path, value, append);
	}
	
	public void set(String[] path, PropertyValue value) {
		this.set(path, value, false);
	}
	
	public void set(String[] path, PropertyValue value, boolean append) {
		IndexedField idxF = new IndexedField(path[0]);
		if (properties!=null) {
			for (Property prop : properties) {
				if (prop.getLabel().equals(idxF.getFieldname())) {
					this.set(prop, path, value, append);
					return;
				}
			}
		}
		Property newProp = new PropertyImpl();
		Optional<PropertyDefinition> pd;
		if (this.getPropertyDefinitions()!=null) {
			pd = this.getPropertyDefinitions().stream().filter(d -> d.getName().equals(idxF.getFieldname())).findFirst();
			if (pd.isPresent()) {
				newProp.setDefinition(pd.get());
			}
		}
		
		newProp.setLabel(idxF.getFieldname());
		this.set(newProp, path, value, append);
		this.add(newProp);
	}
	
	private void set(Property property, String[] path, PropertyValue value, boolean append) {
		IndexedField idxF = new IndexedField(path[0]);
		
		// Path is current level, no index set
		if (path.length==1 && !idxF.hasIndex()) {
			if (append) {
				property.addValue(PropertyDefinitionHelper.transformIfRequiredAndPossible(property.getDefinition(), value));
			} else {
				property.setValue(PropertyDefinitionHelper.transformIfRequiredAndPossible(property.getDefinition(), value));
			}
		} 
		// Path is current level, index set
		else if (path.length==1) {
			if (append) {
				property.addValue(this.getOrCreateValueListWithValue(property, idxF.getIndex(), PropertyDefinitionHelper.transformIfRequiredAndPossible(property.getDefinition(), value)));
			} else {
				property.setValue(this.getOrCreateValueListWithValue(property, idxF.getIndex(), PropertyDefinitionHelper.transformIfRequiredAndPossible(property.getDefinition(), value)));
			}
		} 
		// Path is not current level
		else {
			int idx = idxF.hasIndex() ? idxF.getIndex() : 0;
			
			// Ensure index available
			property.setProperties(this.createOrMergePropertyLists(property.getProperties(), idx));
			if (property.getProperties().get(idx)==null) {
				List<PropertyDefinition> pd = null;
				if(property.getDefinition()!=null && property.getDefinition().isHierarchical()) {
					pd = HierarchicalPropertyDefinition.class.cast(property.getDefinition()).getProperties();
				}
				property.getProperties().set(idx, new PropertyList(pd));
			}
			property.getProperty(idx).set(Arrays.copyOfRange(path, 1, path.length), value);
		}		
	}

	private List<PropertyList> createOrMergePropertyLists(List<PropertyList> source, int index) {
		if (source!=null && source.size()>index+1) {
			return source;
		}
		List<PropertyList> target = new ArrayList<>(index+1);
		for (int i=0; i<=index; i++) {
			if (source!=null && source.size()>i) {
				target.add(source.get(i));
			} else {
				target.add(null);
			}
		}
		return target;
	}
	
	private PropertyValueList getOrCreateValueListWithValue(Property property, int index, PropertyValue value) {
		PropertyValueList valueList;
		if (property.getValue()==null || !PropertyValueList.class.isAssignableFrom(property.getValue().getClass())) {
			valueList = new PropertyValueList();
		} else {
			valueList = PropertyValueList.class.cast(property.getValue());
		}
		valueList.setValues(this.mergeValueListsWithValue(valueList.getValues(), index, value));
		return valueList;				
	}
	
	private List<PropertyValue> mergeValueListsWithValue(List<PropertyValue> source, int index, PropertyValue value) {
		if (source!=null && source.size()>index+1) {
			source.set(index, value);
			return source;
		}
		
		List<PropertyValue> target = new ArrayList<>(index+1);
		for (int i=0; i<=index; i++) {
			if (i==index) {
				target.add(value);
			} else if (source!=null && source.size()>i) {
				target.add(source.get(i));
			} else {
				target.add(null);
			}
		}
		return target;
	}
	
	public void addAll(Collection<? extends Property> properties) {
		this.ensurePropertiesInitialized();
		this.properties.addAll(properties);
	}
	
	public void add(Property property) {
		this.ensurePropertiesInitialized();
		this.properties.add(property);
	}
	
	private void ensurePropertiesInitialized() {
		if (this.properties==null) {
			this.properties = new ArrayList<>();
		}
	}
}
